<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Invoices extends AdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('invoices_model');
        $this->load->model('credit_notes_model');

        $this->app_scripts->add('crm-js', 'assets/js/crm.js');
        $this->app_css->add('item_documents', 'assets/css/item_documents.css');

        // Tailwind 
        $this->app_css->add('defStyle', 'assets/css/tailwind2.css');
        $this->app_css->add('defStyle', 'assets/css/defStyle.css');


        // Rxjs
        $this->app_scripts->add('rxjs', 'https://cdnjs.cloudflare.com/ajax/libs/rxjs/7.4.0/rxjs.umd.min.js');

        $this->app_scripts->add('inputmask-js', 'assets/js/inputmask.js');
        $this->app_scripts->add('inputbinding-js', 'assets/js/inputbinding.js');

        // Main js (to be moved) $this->app_scripts->add('invoice-js', 'assets/js/shared/invoice/invoice.js', 'admin', [], 'module');


        $this->load->model('billing/billing_series_model');
        $this->load->helper('file');
    }

    /* Get all invoices in case user go on index page */
    public function index($id = '')
    {
        $this->list_invoices($id);
    }

    public function import($provider = null, $id = null)
    {
        $data['title'] = _l('import');

        // Load the JSON file with providers
        $providersJson = file_get_contents(INVOICESIMPORT . '/providers.json');
        $data['providers'] = json_decode($providersJson, true);

        // Selected provider@#

        $data['selectedProviderId'] = $id;

        // Imports
        $data['imports'] = [];

        $settings['vat_payer'] = (int)get_option('settings_company_vat_payer');
        $settings['vat_on_collection'] = (int)get_option('settings_company_vat_on_collection');

        if ($settings['vat_on_collection'] == 1) {
            $display_vat_collection = '<p>' . _l('invoice_notice_vat_on_collection') . '</p>';
            $settings['display_vat_collection'] = $display_vat_collection;
        }

        $settings['show_tax_per_item'] = (int)get_option('show_tax_per_item');

        $data['settings'] = $settings;
        $this->load->view('admin/invoices/import/index_template', $data);
    }

    function getProviderById($id)
    {
        $providersJson = file_get_contents(INVOICESIMPORT . '/providers.json');

        // Decode JSON data into an associative array
        $providersData = json_decode($providersJson, true);

        // Iterate through providers to find the one with the target ID
        foreach ($providersData as $provider) {
            if (isset($provider['id']) && $provider['id'] == $id) {
                return $provider;
            }
        }

        // If no provider with the target ID is found, return null or handle as needed
        return null;
    }

    public function uploadImport()
    {

        // First we will make sure we have all directories created
        // if not we will create, somehow the 116 migration fail on some instalations

        // Ensure that the necessary directories exist
        $this->ensure_invoice_folders_exist();

        $selectedProviderId = $_POST['selectedProviderId'];
        $selectedProvider = $this->getProviderById($selectedProviderId);

        // Check if the form was submitted
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Check if a file was uploaded
            if (isset($_FILES['userfile']) && $_FILES['userfile']['error'] === UPLOAD_ERR_OK) {

                // Validate and read the file
                $fileResult = $this->validateAndReadFile($_FILES['userfile'], $selectedProvider);

                if (isset($fileResult['error'])) {
                    // Handle the error
                    echo $fileResult['error'];
                    return; // Stop further execution
                }

                // Define the upload directory
                $uploadDir = INVOICES_IMPORT_FOLDER_IMPORTS;

                // Switch case for file types
                switch ($fileResult['type']) {
                    case 'excel':
                        // Proceed with the valid Excel content
                        $jsonResult = $fileResult['content'];

                        // Generate a unique filename
                        $fileName = uniqid('import_', true) . '_' . $_FILES['userfile']['name'];

                        // Move the uploaded file to the specified directory
                        move_uploaded_file($_FILES['userfile']['tmp_name'], $uploadDir . $fileName);

                        // Process and write Excel import
                        $importId = $this->processAndWriteExcelImport($uploadDir, $fileName, $_FILES['userfile']['type'], $jsonResult, $selectedProviderId);

                        // Optionally, you can redirect or show a success message
                        redirect('admin/invoices/import/success/' . $importId);
                        break;
                    case 'xml':

                        // Proceed with the valid Excel content
                        $jsonResult = $fileResult['content'];

                        // Generate a unique filename
                        $fileName = uniqid('import_', true) . '_' . $_FILES['userfile']['name'];

                        // Move the uploaded file to the specified directory
                        move_uploaded_file($_FILES['userfile']['tmp_name'], $uploadDir . $fileName);

                        // Process and write Excel import
                        $importId = $this->processAndWriteXMLImport($uploadDir, $fileName, $_FILES['userfile']['type'], $jsonResult, $selectedProviderId);

                        redirect('admin/invoices/import/success/' . $importId);
                        break;
                    default:
                        // Redirect to an error page for other file types
                        redirect('invoices/import/error'); // Change the URL accordingly
                        break;
                }
            } else {
                // Handle file upload error
                echo 'File upload error.';
            }
        } else {
            // Handle non-POST request (optional)
            echo 'Invalid request method.';
        }
    }

    public function processImport($importId = null, $batchAmount = 10)
    {

        // Read the content of the excel_data_importid.json file
        $excelDataFile = INVOICES_IMPORT_FOLDER_IMPORTS . 'excel_data_' . $importId . '.json';

        if (file_exists($excelDataFile)) {

            // Read the original content of the file
            $originalData = json_decode(file_get_contents($excelDataFile), true);

            // Read the JSON content
            $excelDataContent = json_decode(file_get_contents($excelDataFile), true);

            // Identify the indices of the items to be processed
            $indicesToProcess = array_keys(array_filter($originalData, function ($item) {
                return isset($item['notfound']) && $item['notfound'] === true && isset($item['processed']) && $item['processed'] === false;
            }));

            // Take a batch of items based on the provided batch amount
            $batchItems = array_intersect_key($originalData, array_flip(array_slice($indicesToProcess, 0, $batchAmount)));

            // Initialize an array to store processed items
            $processedItems = [];

            $firstItem = current($batchItems);

            if (count($batchItems) > 0 && isset($firstItem['cancel_merged_invoices'])) {
                // INVOICE
                $clientDataArray = [];

                foreach ($batchItems as $index => $item) {
                    if ($item['toprocess']) {

                        /**
                         * We will need to check if there is a serie in the system with the serie name
                         * FOUND when we have serie we will check item number and serie latest number
                         * whenever item number is bigger than serie number we update the serie curent_number
                         *
                         * NOT FOUND we insert the serie and add as start number 1 and current_number the item number
                         * $item['s_prefix'] / $item['number']
                         */
                        $serie = null;
                        $serie = $this->checkSerie($item['s_prefix']);

                        // FOUND
                        if ($serie) {

                            if ((int)$item['number'] > (int)$serie['current_number']) {

                                $serieData = [
                                    'current_number' => $item['number']
                                ];
                                $CI =& get_instance();
                                $updateSerie = $CI->billing_series_model->update($serie['id'], $serieData);
                            }

                        } else {
                            // NOT FOUND

                            // Create the arr
                            $serieData = [
                                'type' => 'invoice',
                                'name' => $item['s_prefix'],
                                'start_number' => 0,
                                'current_number' => $item['number'],
                                'description' => 'import'
                            ];

                            $CI =& get_instance();
                            $CI->load->model('billing_series_model');

                            $insertSerie = $CI->billing_series_model->add($serieData);
                        }

                        $item['status'] = 2;
                        $invoice = $this->invoices_model->add($item, false, true);

                        if ($invoice) {
                            $item['processed'] = true;
                            $item['toprocess'] = false;
                            $item['processedID'] = $invoice;
                        }
                    }
                    // Add the processed item to the array
                    $processedItems[$index] = $item;
                }

            } else {

                /// CLIENT
                // Loop through the content and process each entry
                foreach ($batchItems as $index => $clientData) {
                    if ($clientData['toprocess']) {
                        $clientCountry = 0;
                        if (isset($clientData['Tara'])) {
                            $clientCountry = $this->clients_model->getCountry($clientData['Tara']);
                        }


                        $firstname = $this->extractNamePart('firstname', $clientData['Pers contact']);
                        $lastname = $this->extractNamePart('lastname', $clientData['Pers contact']);

                        $addclient = $this->clients_model->add([
                            'vat' => isset($clientData['CIF']) ? $clientData['CIF'] : '',
                            'company' => isset($clientData['Denumire client']) ? $clientData['Denumire client'] : '',
                            'trade_number' => isset($clientData['Reg com']) ? $clientData['Reg com'] : '',
                            'address' => isset($clientData['Adresa']) ? $clientData['Adresa'] : '',
                            'city' => isset($clientData['Localitate']) ? $clientData['Localitate'] : '',
                            'state' => isset($clientData['Judet']) ? $clientData['Judet'] : '',
                            'bank' => isset($clientData['Banca']) ? $clientData['Banca'] : '',
                            'iban' => isset($clientData['Iban']) ? $clientData['Iban'] : '',
                            'country' => isset($clientCountry->country_id) ? $clientCountry->country_id : '',
                            'organization_email' => isset($clientData['Email']) ? $clientData['Email'] : '',
                            'phonenumber' => isset($clientData['Telefon']) ? $clientData['Telefon'] : '',
                            'firstname' => $firstname,
                            'lastname' => $lastname,
                        ], $firstname ? true : false, true);

                        if ($addclient) {
                            $clientData['processed'] = true;
                            $clientData['toprocess'] = false;
                            $clientData['processedID'] = $addclient;
                        }
                    }

                    // Add the processed item to the array
                    $processedItems[$index] = $clientData;
                }
            }

            $this->replaceExcelDataContent($importId, $processedItems);
        } else {
            // Handle the case when the file doesn't exist
            // ...
        }


        // Redirect back to the same page
        //redirect('admin/invoices/import/success/' . $importId);
    }

    /**
     * Series
     */

    function checkSerie($serie)
    {

        $this->db->where('type', 'invoice');
        $this->db->where('name', $serie);
        $row = $this->db->get(db_prefix() . 'billing_series')->row_array();

        if ($row) {
            return $row;
        } else {
            return null;
        }
    }

    public function success($importId = null)
    {
        // Validate and sanitize the import ID if needed
        // ...

        // Pass the import ID to the view
        $data['importId'] = $importId;

        // Load the import data from imports.json
        $data['importData'] = $this->getImportDataById($importId);

        $data['fileDataContent'] = $this->getFileDataContent($data['importData']);

        // Update import data with statistics
        $this->updateImportDataStats($data['importData'], $data['fileDataContent']);

        // Load the import data from imports.json
        $data['importData'] = $this->getImportDataById($importId);

        // Load the view
        $data['title'] = _l('import');
        $this->load->view('admin/invoices/import/provider_import_success_template', $data);
    }

    public function batchprocess($importId = null)
    {
        $batchAmount = 40;

        // Generate stats to check the current status
        $data['stats'] = $this->generateStats($importId, $batchAmount);
        $data['title'] = _l('import');

        // Check if there are items left to process
        if ($data['stats']['itemsLeftToProcess'] > 0) {
            // Process
            $this->processImport($importId, $batchAmount);

            // Display the initial processing info
            $data['progressInfo'] = _l('processing_status_running');
        } else {
            // No items left to process, display a message
            $data['progressInfo'] = _l('processing_status_finished');
        }

        $this->load->view('admin/invoices/import/batch_process', $data);
    }


    private function replaceExcelDataContent($importId, $processedItems)
    {
        // Get the file path based on the import ID
        $uploadDir = INVOICES_IMPORT_FOLDER_IMPORTS;
        $excelDataFilePath = $uploadDir . 'excel_data_' . $importId . '.json';

        // Check if the file exists
        if (file_exists($excelDataFilePath)) {
            // Read the original content of the file
            $originalData = json_decode(file_get_contents($excelDataFilePath), true);

            // Update only the processed items in the original data
            foreach ($processedItems as $index => $processedItem) {
                $originalData[$index] = $processedItem;
            }

            // Convert the updated data array back to JSON
            $updatedJsonData = json_encode($originalData, JSON_PRETTY_PRINT);

            // Replace the content of the file with the updated content
            file_put_contents($excelDataFilePath, $updatedJsonData);

            // Return true on success
            return true;
        }

        // Return false if the file doesn't exist
        return false;
    }

    public function generateStats($importId = null, $batchAmount = 10)
    {
        // Ensure importId is provided
        if ($importId === null) {
            // Handle the case when importId is not provided
            return ['error' => 'Import ID is missing.'];
        }

        // Ensure batchAmount is a positive integer
        if (!is_int($batchAmount) || $batchAmount <= 0) {
            return ['error' => 'Invalid batch amount.'];
        }

        // Construct the file path
        $uploadDir = INVOICES_IMPORT_FOLDER_IMPORTS;
        $excelDataFilePath = $uploadDir . 'excel_data_' . $importId . '.json';

        // Check if the file exists
        if (!file_exists($excelDataFilePath)) {
            // Handle the case when the file doesn't exist
            return ['error' => 'Data file not found.'];
        }

        // Read the JSON content
        $excelDataContent = json_decode(file_get_contents($excelDataFilePath), true);

        // Initialize counters for statistics
        $totalItems = count($excelDataContent);
        $foundAndProcessed = 0;
        $foundAndNotProcessed = 0;
        $processed = 0;
        $notfound = 0;

        // Iterate through the data and update counters
        foreach ($excelDataContent as $item) {
            if (isset($item['notfound']) && $item['notfound'] === true) {
                if (isset($item['processed']) && $item['processed'] === true) {
                    // Item found and processed
                    $foundAndProcessed++;
                } else {
                    // Item found but not processed
                    $foundAndNotProcessed++;
                }
            }

            if (isset($item['processed']) && $item['processed'] === true) {
                // Item found and processed
                $processed++;
            }

            if (isset($item['notfound']) && $item['notfound'] === true) {
                // Item found and processed
                $notfound++;
            }
        }


        // Calculate how many items are left to process
        $itemsLeftToProcess = max(0, $notfound - $processed);

        // Calculate the number of batches
        $totalBatches = ceil($notfound / $batchAmount);

        // Calculate how many batches are left to process
        $batchesProcessed = ceil($processed / $batchAmount);
        $batchesLeft = max(0, $totalBatches - $batchesProcessed);

        // Calculate the current batch being processed
        $currentBatch = min($batchesProcessed + 1, $totalBatches);

        // Return the statistics as an associative array
        return [
            'totalItems' => $totalItems,
            'totalBatches' => $totalBatches,
            'batchesLeft' => $batchesLeft,
            'currentBatch' => $currentBatch,
            'foundAndProcessed' => $foundAndProcessed,
            'foundAndNotProcessed' => $foundAndNotProcessed,
            'processingNow' => $batchAmount,
            'itemsLeftToProcess' => $itemsLeftToProcess,
            'processed' => $processed,
            'notfound' => $notfound
        ];
    }

    private function updateImportDataStats(&$importData, $fileDataContent)
    {

        if (!$importData || !isset($importData['stats'])) {
            return; // No import data or stats not found
        }

        $found = 0;
        $noFound = 0;
        $clientDataArray = [];

        if ($importData['fileDataType'] && $importData['fileDataType'] == 'invoices') {
            foreach ($fileDataContent as $clientData) {
                // we will need to search in database invoices with this serie and number
                $sql = 'SELECT * FROM ' . db_prefix() . 'invoices WHERE (prefix="' . $clientData['s_prefix'] . '" AND number="' . $clientData['number'] . '")';
                $invoices = $this->db->query($sql)->result_array();


                if ($invoices) {
                    // Invoice found
                    $found++;
                    $clientData['found'] = true;
                    $clientData['notfound'] = false;
                } else {
                    // Invoice not found
                    $noFound++;
                    $clientData['found'] = false;
                    $clientData['notfound'] = true;
                }

                // Additional attributes
                $clientData['processed'] = false; // Mark as not processed initially
                $clientData['toprocess'] = $clientData['notfound']; // Set to true if not found, false otherwise

                // Add $clientData to the array
                $clientDataArray[] = $clientData;

            }

            $this->replaceExcelDataContent($importData['id'], $clientDataArray);


            // Update stats in the import data
            $importData['stats']['found'] = $found;
            $importData['stats']['noFound'] = $noFound;

            // Update the existing data back to imports.json
            $importsFilePath = INVOICES_IMPORT_FOLDER_IMPORTS . 'imports.json';
            $existingData = json_decode(file_get_contents($importsFilePath), true);

            foreach ($existingData as &$existingImport) {
                if ($existingImport['id'] === $importData['id']) {
                    $existingImport = $importData;
                    break;
                }
            }

            // Write the updated data back to imports.json
            file_put_contents($importsFilePath, json_encode($existingData));
        }

        if ($importData['fileDataType'] && $importData['fileDataType'] == 'client') {

            foreach ($fileDataContent as $clientData) {
                $this->db->where('vat', $clientData['CIF']);
                $client = $this->db->get(db_prefix() . 'clients')->row();

                if ($client) {
                    // Client found
                    $found++;
                    $clientData['found'] = true;
                    $clientData['notfound'] = false;
                } else {
                    // Client not found
                    $noFound++;
                    $clientData['found'] = false;
                    $clientData['notfound'] = true;
                }

                // Additional attributes
                $clientData['processed'] = false; // Mark as not processed initially
                $clientData['toprocess'] = $clientData['notfound']; // Set to true if not found, false otherwise

                // Add $clientData to the array
                $clientDataArray[] = $clientData;
            }

            $this->replaceExcelDataContent($importData['id'], $clientDataArray);


            // Update stats in the import data
            $importData['stats']['found'] = $found;
            $importData['stats']['noFound'] = $noFound;

            // Update the existing data back to imports.json
            $importsFilePath = INVOICES_IMPORT_FOLDER_IMPORTS . 'imports.json';
            $existingData = json_decode(file_get_contents($importsFilePath), true);

            foreach ($existingData as &$existingImport) {
                if ($existingImport['id'] === $importData['id']) {
                    $existingImport = $importData;
                    break;
                }
            }

            // Write the updated data back to imports.json
            file_put_contents($importsFilePath, json_encode($existingData));
        }
    }

    private function getExcelDataContentById($importId)
    {
        $excelDataFileName = 'excel_data_' . $importId . '.json';
        $excelDataFilePath = INVOICES_IMPORT_FOLDER_IMPORTS . $excelDataFileName;

        if (file_exists($excelDataFilePath)) {
            $excelDataContent = file_get_contents($excelDataFilePath);
            return json_decode($excelDataContent, true);
        }

        return [];
    }

    private function getFileDataContent($importData)
    {
        if (!$importData || !isset($importData['fileDataName'])) {
            return null; // No import data or fileDataName not found
        }

        $fileDataPath = INVOICES_IMPORT_FOLDER_IMPORTS . $importData['fileDataName'];

        if (!file_exists($fileDataPath)) {
            return null; // File not found
        }

        // Read and return the content of the file
        $fileContent = file_get_contents($fileDataPath);
        return json_decode($fileContent, true);
    }

    private function getImportDataById($importId)
    {
        // Read the existing data from imports.json
        $importsFilePath = INVOICES_IMPORT_FOLDER_IMPORTS . 'imports.json';

        $existingData = json_decode(file_get_contents($importsFilePath), true);

        // Find the import data with the specified ID
        foreach ($existingData as $import) {
            if ($import['id'] == $importId) {
                return $import;
            }
        }

        // If import ID is not found, you may handle this case accordingly (e.g., show an error)
        return null;
    }

    private function processAndWriteXMLImport($uploadDir, $fileName, $fileType, $jsonResult, $selectedProviderId)
    {

        // Update the imports.json file with information about the imported file
        $importsFilePath = $uploadDir . 'imports.json';

        // Get the highest existing ID
        $nextId = $this->getHighestImportId($importsFilePath) + 1;

        // Calculate statistics (example: count of items)
        $stats = [
            'items' => count($jsonResult),
            // Add more statistics as needed
        ];


        // client for excel extension case and invoices for xml extension case only for SmartBill provider
        $fileDataType = ($selectedProviderId == 1 && in_array(strtolower(pathinfo($fileName, PATHINFO_EXTENSION)), ['xls', 'xlsx'])) ? 'client' : 'invoices';

        // Create a custom file name for the Excel data file
        $fileDataName = 'excel_data_' . $nextId . '.json';

        $importData = [
            'id' => $nextId,
            'fileName' => $fileName,
            'fileExtension' => pathinfo($fileName, PATHINFO_EXTENSION),
            'fileType' => $fileType,
            'fileSize' => $_FILES['userfile']['size'],
            'date' => date('d-m-Y H:i', time()),
            'stats' => $stats, // Include the statistics,
            'filePath' => $uploadDir . $fileName, // Add the file path
            'fileDataName' => $fileDataName, // Add the custom file name for Excel data
            'fileDataType' => $fileDataType, // Add the file data type
            'providerId' => $selectedProviderId, // Add the provider ID
        ];

        // Read the existing data from imports.json
        $existingData = json_decode(file_get_contents($importsFilePath), true);

        // Add the new import data to the array
        $existingData[] = $importData;

        // Write the updated data back to imports.json
        file_put_contents($importsFilePath, json_encode($existingData));

        // Write the Excel data to a separate JSON file with the import ID as the filename
        $excelDataFilePath = $uploadDir . $fileDataName;
        file_put_contents($excelDataFilePath, json_encode($jsonResult));

        return $nextId;
    }

    private function processAndWriteExcelImport($uploadDir, $fileName, $fileType, $jsonResult, $selectedProviderId)
    {

        // Update the imports.json file with information about the imported file
        $importsFilePath = $uploadDir . 'imports.json';

        // Get the highest existing ID
        $nextId = $this->getHighestImportId($importsFilePath) + 1;

        // Calculate statistics (example: count of items)
        $stats = [
            'items' => count($jsonResult),
            // Add more statistics as needed
        ];

        // client for excel extension case and invoices for xml extension case only for SmartBill provider
        $fileDataType = ($selectedProviderId == 1 && in_array(strtolower(pathinfo($fileName, PATHINFO_EXTENSION)), ['xls', 'xlsx'])) ? 'client' : 'invoices';

        // Create a custom file name for the Excel data file
        $fileDataName = 'excel_data_' . $nextId . '.json';

        $importData = [
            'id' => $nextId,
            'fileName' => $fileName,
            'fileExtension' => pathinfo($fileName, PATHINFO_EXTENSION),
            'fileType' => $fileType,
            'fileSize' => $_FILES['userfile']['size'],
            'date' => date('d-m-Y H:i', time()),
            'stats' => $stats, // Include the statistics,
            'filePath' => $uploadDir . $fileName, // Add the file path
            'fileDataName' => $fileDataName, // Add the custom file name for Excel data
            'fileDataType' => $fileDataType, // Add the file data type
            'providerId' => $selectedProviderId, // Add the provider ID
        ];

        // Read the existing data from imports.json
        $existingData = json_decode(file_get_contents($importsFilePath), true);

        // Add the new import data to the array
        $existingData[] = $importData;

        // Write the updated data back to imports.json
        file_put_contents($importsFilePath, json_encode($existingData));

        // Write the Excel data to a separate JSON file with the import ID as the filename
        $excelDataFilePath = $uploadDir . $fileDataName;


        // Function to recursively apply utf8_encode to array elements
        $utf8EncodeRecursive = function ($item) use (&$utf8EncodeRecursive) {
            return is_array($item) ? array_map($utf8EncodeRecursive, $item) : mb_convert_encoding($item, 'UTF-8', 'UTF-8');
        };

        $utf8EncodedArray = array_map($utf8EncodeRecursive, $jsonResult);

        // Now encode the entire array as JSON
        $jsonData = json_encode($utf8EncodedArray, JSON_PRETTY_PRINT);

        if ($jsonData === false) {
            echo 'JSON encoding error: ' . json_last_error_msg();
        } else {

            // Write the JSON data to the file
            file_put_contents($excelDataFilePath, $jsonData);

            if (file_exists($excelDataFilePath)) {
                return $nextId;
            } else {
                echo 'Error writing JSON data to ' . $excelDataFilePath;
            }
        }
    }

    private function validateAndReadFile($file, $providerData)
    {
        $allowedExtensions = ['xls', 'xlsx', 'xml'];

        $fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);

        if (!in_array(strtolower($fileExtension), $allowedExtensions)) {
            // Invalid file type
            return ['error' => 'Invalid file type. Please upload an Excel (xls, xlsx) or XML file.'];
        }

        // Define the upload directory
        $uploadDir = INVOICES_IMPORT_FOLDER_IMPORTS;

        // Generate a unique filename
        $fileName = uniqid('uploaded_', true) . '.' . $fileExtension;

        // Move the uploaded file to the specified directory
        move_uploaded_file($file['tmp_name'], $uploadDir . $fileName);

        // Perform additional validations if needed

        // Return file type and the actual file path
        switch (strtolower($fileExtension)) {
            case 'xls':
            case 'xlsx':
                // Assuming you have a function to convert Excel to JSON
                $jsonResult = $this->excelToJson($uploadDir . $fileName);

                // Get the client structure map from the provider data
                $clientStructureMap = $providerData['clientStructureMap'];

                // Validate JSON structure using the client structure map
                $this->validateJsonStructure($jsonResult, $clientStructureMap);

                return ['type' => 'excel', 'content' => $jsonResult];
            case 'xml':
                // Assuming you have a function to convert Excel to JSON
                $jsonResult = $this->readXmlFile($uploadDir . $fileName);

                // Get the client structure map from the provider data
                $clientStructureMap = $providerData['invoiceStructureMap'];

                // Validate JSON structure using the client structure map
                $this->validateJsonStructure($jsonResult, $clientStructureMap);

                // make opur structure
                $jsonResult = $this->makeInvoiceInternalStructure($jsonResult);

                // Assuming you have a function to read XML content
                return ['type' => 'xml', 'content' => $jsonResult];
            default:
                return ['error' => 'Invalid file type. Please upload an Excel (xls, xlsx) or XML file.'];
        }
    }

    private function makeInvoiceInternalStructure($jsonResult)
    {


        $internalArray = [];
        $this->load->model('currencies_model');

        //echo json_encode($jsonResult, JSON_PRETTY_PRINT);die;
        //var_dump($jsonResult);die;

        // Iterate through each item in the $jsonResult array
        foreach ($jsonResult as $item) {

            // Create a new object to store internal structure data
            $internalObject = new stdClass();

            //Find client by CIF else skip this 

            $this->db->where('vat', $item['ClientCIF']);
            $client = $this->db->get(db_prefix() . 'clients')->row();

            // Check if $client is empty, and skip the current iteration if true
            if (empty($client)) {
                continue;
            }

            $facturaData = new DateTime($item['FacturaData']);
            $facturaScadenta = new DateTime($item['FacturaScadenta']);

            // Split FacturaNumar into serie and number
            $facturaNumar = $item['FacturaNumar'];
            preg_match('/^(.*?)(\d+)$/', $facturaNumar, $matches);

            $currency = $this->currencies_model->get_by_name($item['FacturaMoneda'])->id;

            // Add data to the internal object as needed
            $internalObject->cancel_merged_invoices = "on";
            $internalObject->clientid = $client->userid;
            $internalObject->show_shipping_on_invoice = "on";
            $internalObject->date = $facturaData->format('d-m-Y');
            $internalObject->duedate = $facturaScadenta->format('d-m-Y');
            $internalObject->s_prefix = $matches[1];
            $internalObject->number = $matches[2];
            $internalObject->currency = $currency;
            $internalObject->invoice_language = "1";

            $internalObject->allowed_payment_modes = [
                "1",
                "2",
                "netopia",
                "mollie",
                "paypal_braintree",
                "paypal_checkout",
                "stripe"
            ];
            $internalObject->recurring = "0";
            $internalObject->repeat_every_custom = "1";
            $internalObject->repeat_type_custom = "day";
            $internalObject->discount_type = "before_tax";
            $internalObject->quantity = "1";
            $internalObject->invoice_item_currency = $currency;
            $internalObject->invoice_item_currency_rate = "1";
            $internalObject->newitems = [];

            foreach ($item['items'] as $invoiceItem) {
                // Create a new object to represent each invoice item
                $invoiceItemObject = new stdClass();

                // Assign properties for each invoice item
                if (!isset($invoiceItem['LinieNrCrt'])) {
                    die;
                    //echo json_encode($item['items'], JSON_PRETTY_PRINT);die;
                }
                $invoiceItemObject->order = $invoiceItem['LinieNrCrt'];
                $invoiceItemObject->description = $invoiceItem['Descriere'];
                $invoiceItemObject->long_description = $invoiceItem['Descriere'];
                $invoiceItemObject->sku = "";
                $invoiceItemObject->unit = $invoiceItem['UM'];
                $invoiceItemObject->qty = $invoiceItem['Cantitate'];

                // Calculate the tax value
                $taxValue = round($invoiceItem['CotaTVA'], 2);

                // Create the "taxname" array
                $taxname = ["TVA {$taxValue}%|{$invoiceItem['ProcTVA']}"];
                $invoiceItemObject->taxname = $taxname;

                $invoiceItemObject->currency = $currency;
                $invoiceItemObject->currency_rate = "1";
                $invoiceItemObject->rate = $invoiceItem['Pret'];
                $invoiceItemObject->price_converted = $invoiceItem['Pret'];

                // Push the invoice item object to the newitems array
                $internalObject->newitems[] = $invoiceItemObject;
            }
            $internalObject->subtotal = $item['Sumar']['Total'];
            $internalObject->discount_percent = "0";
            $internalObject->discount_total = "0";
            $internalObject->total = $item['Sumar']['TotalValoare'];
            $internalObject->clientnote = $item['Observatii']['txtObservatii'];

            // Push the internal object to the internal array
            $internalArray[] = $internalObject;
        }

        return $internalArray;
    }

    private function validateJsonStructure($jsonResult, $expectedStructure)
    {

        // Get the keys of the first row in the JSON data
        $actualKeys = array_keys($jsonResult[0]);

        //var_dump($actualKeys);die;

        // Check if the actual keys match the expected structure
        if ($actualKeys !== $expectedStructure) {
            // Prepare error data
            $message = 'import_file_structure_error';

            // Set the error message in the session
            $this->session->set_flashdata('errorMessage', $message);

            // Redirect to the error page with error data
            redirect('admin/invoices/errorpage');
        }
    }

    public function errorpage($errorMessage = null)
    {
        $data['title'] = _l('invoices');

        // Pass the error message to the view
        $data['errorMessage'] = $this->session->flashdata('errorMessage');

        // Load the error view
        $this->load->view('admin/invoices/import/provider_import_error_template', $data);
    }

    public function showError($errorMessage = null)
    {
        $data['errorMessage'] = $errorMessage;
        // Add any other data you want to pass to the error view

        // Load the error view
        $this->load->view('error_view', $data);
    }


    // Function to read XML content
    private function readXmlFile($filePath)
    {
        // Check if the file exists
        if (!file_exists($filePath)) {
            // Handle the case where the file doesn't exist
            return json_encode(['error' => 'File not found']);
        }

        // Read XML content from the file
        $xml = simplexml_load_file($filePath);

        // Check if the XML is valid
        if ($xml === false) {
            // Handle the case where the XML is not valid
            return json_encode(['error' => 'Invalid XML']);
        }

        // Convert SimpleXMLElement to array
        $dataArray = $this->xmlToArray($xml);

        // Dump the array for better viewing during development


        return $this->makeInvoiceStructure($dataArray);
    }

    private function xmlToArray($xml)
    {
        $array = [];

        foreach ($xml->children() as $factura) {
            $facturaData = [];

            foreach ($factura->children() as $sectionName => $section) {
                $facturaData[$sectionName] = $this->parseSection($section);
            }

            $array[] = $facturaData;
        }

        return $array;
    }

    private function parseSection($section)
    {
        $data = [];

        foreach ($section->children() as $element) {
            if ($element->count() === 0) {
                $data[$element->getName()] = (string)$element;
            } else {
                $data[$element->getName()][] = $this->parseSection($element);
            }
        }

        return $data;
    }

    function makeInvoiceStructure($data)
    {


        // Decode the JSON data
        //$data = json_decode($jsonData, true);
//echo json_encode($data, JSON_PRETTY_PRINT);die;
        $newArray = [];

        // Iterate through each item in the "Factura" array
        foreach ($data as $factura) {
            // Create a new object to store "Antet" content
            $antetObject = $factura['Antet'];
            // Add more properties as needed

            // Create a new object to store "Detalii" content
            $detaliiObject = new stdClass();
            $detaliiObject->items = $factura['Detalii']['Continut'][0]['Linie'];


            // Add more properties as needed

            // Combine "Antet" content and "Detalii" object into a single object
            $newObject = $antetObject;
            $newObject['items'] = $detaliiObject->items;

            // Add the content of "<Sumar>" to the object
            $newObject['Sumar'] = $factura['Sumar'];

            // Add the content of "<Observatii>" to the object
            $newObject['Observatii'] = $factura['Observatii'];

            // Add the combined object to the new array
            $newArray[] = $newObject;
        }
        // Return the array
        return $newArray;
    }

    // Function to get the highest ID from the existing imports
    private function getHighestImportId($importsFilePath)
    {
        $existingData = json_decode(file_get_contents($importsFilePath), true);

        if (!empty($existingData)) {
            $maxId = max(array_column($existingData, 'id'));
            return $maxId;
        }

        return 0; // Return 0 if the array is empty
    }


    /* List all invoices datatables */
    public function list_invoices($id = '')
    {
        if (!has_permission('invoices', '', 'view')
            && !has_permission('invoices', '', 'view_own')
            && get_option('allow_staff_view_invoices_assigned') == '0') {
            access_denied('invoices');
        }

        close_setup_menu();

        $this->load->model('payment_modes_model');
        $data['payment_modes'] = $this->payment_modes_model->get('', [], true);
        $data['invoiceid'] = $id;
        $data['title'] = _l('invoices');
        $data['invoices_years'] = $this->invoices_model->get_invoices_years();
        $data['invoices_sale_agents'] = $this->invoices_model->get_sale_agents();
        $data['invoices_statuses'] = $this->invoices_model->get_statuses();
        $data['bodyclass'] = 'invoices-total-manual no-calculate-total';

        $settings['vat_payer'] = (int)get_option('settings_company_vat_payer');
        $settings['vat_on_collection'] = (int)get_option('settings_company_vat_on_collection');
        if ($settings['vat_on_collection'] == 1) {
            $display_vat_collection = '<p>' . _l('invoice_notice_vat_on_collection') . '</p>';
            $settings['display_vat_collection'] = $display_vat_collection;
        }
        $settings['show_tax_per_item'] = (int)get_option('show_tax_per_item');
        $data['invoice_settings'] = get_option('invoice_settings');
        $data['invoice_settings'] = json_decode($data['invoice_settings'], true);

        $data['settings'] = $settings;

        $this->load->view('admin/invoices/manage', $data);
    }

    /* List all recurring invoices */
    public function recurring($id = '')
    {
        if (!has_permission('invoices', '', 'view')
            && !has_permission('invoices', '', 'view_own')
            && get_option('allow_staff_view_invoices_assigned') == '0') {
            access_denied('invoices');
        }

        close_setup_menu();

        $this->load->model('payment_modes_model');
        $data['payment_modes'] = $this->payment_modes_model->get('', [], true);
        $data['invoiceid'] = $id;
        $data['title'] = _l('invoices_list_recurring');
        $data['invoices_years'] = $this->invoices_model->get_invoices_years();
        $data['invoices_sale_agents'] = $this->invoices_model->get_sale_agents();
        $data['invoices_statuses'] = $this->invoices_model->get_statuses();
        $data['bodyclass'] = 'invoices-total-manual';

        $settings['vat_payer'] = (int)get_option('settings_company_vat_payer');
        $settings['vat_on_collection'] = (int)get_option('settings_company_vat_on_collection');
        if ($settings['vat_on_collection'] == 1) {
            $display_vat_collection = '<p>' . _l('invoice_notice_vat_on_collection') . '</p>';
            $settings['display_vat_collection'] = $display_vat_collection;
        }
        $settings['show_tax_per_item'] = (int)get_option('show_tax_per_item');
        $data['invoice_settings'] = get_option('invoice_settings');
        $data['invoice_settings'] = json_decode($data['invoice_settings'], true);

        $data['settings'] = $settings;

        $this->load->view('admin/invoices/recurring/list', $data);
    }

    public function table($clientid = '')
    {
        if (!has_permission('invoices', '', 'view')
            && !has_permission('invoices', '', 'view_own')
            && get_option('allow_staff_view_invoices_assigned') == '0') {
            ajax_access_denied();
        }

        $this->load->model('payment_modes_model');
        $data['payment_modes'] = $this->payment_modes_model->get('', [], true);

        $this->app->get_table_data(($this->input->get('recurring') ? 'recurring_invoices' : 'invoices'), [
            'clientid' => $clientid,
            'data' => $data,
        ]);
    }

    public function client_change_data($customer_id, $current_invoice = '')
    {
        if ($this->input->is_ajax_request()) {
            $this->load->model('projects_model');
            $data = [];
            $data['billing_shipping'] = $this->clients_model->get_customer_billing_and_shipping_details($customer_id);
            $data['client_currency'] = $this->clients_model->get_customer_default_currency($customer_id);

            $data['customer_has_projects'] = customer_has_projects($customer_id);
            $data['billable_tasks'] = $this->tasks_model->get_billable_tasks($customer_id);

            if ($current_invoice != '') {
                $this->db->select('status');
                $this->db->where('id', $current_invoice);
                $current_invoice_status = $this->db->get(db_prefix() . 'invoices')->row()->status;
            }

            $_data['invoices_to_merge'] = !isset($current_invoice_status) || (isset($current_invoice_status) && $current_invoice_status != Invoices_model::STATUS_CANCELLED) ? $this->invoices_model->check_for_merge_invoice($customer_id, $current_invoice) : [];

            $data['merge_info'] = $this->load->view('admin/invoices/merge_invoice', $_data, true);

            $this->load->model('currencies_model');

            $__data['expenses_to_bill'] = !isset($current_invoice_status) || (isset($current_invoice_status) && $current_invoice_status != Invoices_model::STATUS_CANCELLED) ? $this->invoices_model->get_expenses_to_bill($customer_id) : [];

            $data['expenses_bill_info'] = $this->load->view('admin/invoices/bill_expenses', $__data, true);
            echo json_encode($data);
        }
    }

    public function get_seria()
    {

        $seria = $this->input->post('seria');
        $date = $this->input->post('date');
        $seria_data = $this->invoices_model->get_seria($seria);
        $seria_data['next_number'] = ((int)$seria_data['current_number'] + 1);
        $invoice = $this->invoices_model->get_last_invoice_from_seria($seria);

        if ($invoice) {
            $date_format = get_option('dateformat');
            $date_format = explode('|', $date_format);
            $date_format = $date_format[0];

            $last_date = date($date_format, strtotime($invoice['date']));

            $date_l = DateTime::createFromFormat($date_format, $last_date)->format('Y-m-d');
            $date_d = DateTime::createFromFormat($date_format, $date)->format('Y-m-d');

            if (($date_d) < ($date_l)) {
                $seria_data['last_date'] = $last_date;
            }
        }

        echo json_encode($seria_data);
        die;
    }

    public function update_number_settings($id)
    {
        $response = [
            'success' => false,
            'message' => '',
        ];
        if (has_permission('invoices', '', 'edit')) {
            $affected_rows = 0;

            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'invoices', [
                'prefix' => $this->input->post('prefix'),
            ]);
            if ($this->db->affected_rows() > 0) {
                $affected_rows++;
            }

            if ($affected_rows > 0) {
                $response['success'] = true;
                $response['message'] = _l('updated_successfully', _l('invoice'));
            }
        }
        echo json_encode($response);
        die;
    }

    public function validate_invoice_number()
    {
        $isedit = $this->input->post('isedit');
        $number = $this->input->post('number');
        $date = $this->input->post('date');
        $original_number = $this->input->post('original_number');
        $number = trim($number);
        $number = ltrim($number, '0');

        if ($isedit == 'true') {
            if ($number == $original_number) {
                echo json_encode(true);
                die;
            }
        }

        if (total_rows(db_prefix() . 'invoices', [
                'YEAR(date)' => date('Y', strtotime(to_sql_date($date))),
                'number' => $number,
                'status !=' => Invoices_model::STATUS_DRAFT,
            ]) > 0) {
            echo 'false';
        } else {
            echo 'true';
        }
    }

    public function validate_invoice_number_seria()
    {
        $isedit = $this->input->post('isedit');
        $number = $this->input->post('number');
        $seria = $this->input->post('seria');
        $date = $this->input->post('date');
        $original_number = $this->input->post('original_number');
        $number = trim($number);
        $number = ltrim($number, '0');

        if ($isedit == 'true') {
            if ($number == $original_number) {
                echo json_encode(true);
                die;
            }
        }

        if (total_rows(db_prefix() . 'billing_series', [
                //'YEAR(date)' => date('Y', strtotime(to_sql_date($date))),
                'name' => $seria,
                'current_number >=' => $number,

            ]) > 0) {
            echo 'false';
        } else {
            echo 'true';
        }
    }


    public function add_note($rel_id)
    {
        if ($this->input->post() && user_can_view_invoice($rel_id)) {
            $this->misc_model->add_note($this->input->post(), 'invoice', $rel_id);
            echo $rel_id;
        }
    }

    public function get_notes($id)
    {
        if (user_can_view_invoice($id)) {
            $data['notes'] = $this->misc_model->get_notes($id, 'invoice');
            $this->load->view('admin/includes/sales_notes_template', $data);
        }
    }

    public function pause_overdue_reminders($id)
    {
        if (has_permission('invoices', '', 'edit')) {
            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'invoices', ['cancel_overdue_reminders' => 1]);
        }
        redirect(admin_url('invoices/list_invoices/' . $id));
    }

    public function resume_overdue_reminders($id)
    {
        if (has_permission('invoices', '', 'edit')) {
            $this->db->where('id', $id);
            $this->db->update(db_prefix() . 'invoices', ['cancel_overdue_reminders' => 0]);
        }
        redirect(admin_url('invoices/list_invoices/' . $id));
    }

    public function mark_as_cancelled($id)
    {
        if (!has_permission('invoices', '', 'edit') && !has_permission('invoices', '', 'create')) {
            access_denied('invoices');
        }

        $success = $this->invoices_model->mark_as_cancelled($id);

        if ($success) {
            set_alert('success', _l('invoice_marked_as_cancelled_successfully'));
        }

        redirect(admin_url('invoices/list_invoices/' . $id));
    }

    public function unmark_as_cancelled($id)
    {
        if (!has_permission('invoices', '', 'edit') && !has_permission('invoices', '', 'create')) {
            access_denied('invoices');
        }
        $success = $this->invoices_model->unmark_as_cancelled($id);
        if ($success) {
            set_alert('success', _l('invoice_unmarked_as_cancelled'));
        }
        redirect(admin_url('invoices/list_invoices/' . $id));
    }

    public function copy($id)
    {
        if (!$id) {
            redirect(admin_url('invoices'));
        }
        if (!has_permission('invoices', '', 'create')) {
            access_denied('invoices');
        }
        $new_id = $this->invoices_model->copy($id);
        if ($new_id) {
            set_alert('success', _l('invoice_copy_success'));
            redirect(admin_url('invoices/invoice/' . $new_id));
        } else {
            set_alert('success', _l('invoice_copy_fail'));
        }
        redirect(admin_url('invoices/invoice/' . $id));
    }

    public function get_merge_data($id)
    {
        $invoice = $this->invoices_model->get($id);
        $cf = get_custom_fields('items');

        $i = 0;

        foreach ($invoice->items as $item) {
            $invoice->items[$i]['taxname'] = get_invoice_item_taxes($item['id']);
            $invoice->items[$i]['long_description'] = clear_textarea_breaks($item['long_description']);
            $this->db->where('item_id', $item['id']);
            $rel = $this->db->get(db_prefix() . 'related_items')->result_array();
            $item_related_val = '';
            $rel_type = '';
            foreach ($rel as $item_related) {
                $rel_type = $item_related['rel_type'];
                $item_related_val .= $item_related['rel_id'] . ',';
            }
            if ($item_related_val != '') {
                $item_related_val = substr($item_related_val, 0, -1);
            }
            $invoice->items[$i]['item_related_formatted_for_input'] = $item_related_val;
            $invoice->items[$i]['rel_type'] = $rel_type;

            $invoice->items[$i]['custom_fields'] = [];

            foreach ($cf as $custom_field) {
                $custom_field['value'] = get_custom_field_value($item['id'], $custom_field['id'], 'items');
                $invoice->items[$i]['custom_fields'][] = $custom_field;
            }
            $i++;
        }
        echo json_encode($invoice);
    }

    public function get_bill_expense_data($id)
    {
        $this->load->model('expenses_model');
        $expense = $this->expenses_model->get($id);

        $expense->qty = 1;
        $expense->long_description = clear_textarea_breaks($expense->description);
        $expense->description = $expense->name;
        $expense->rate = $expense->amount;
        if ($expense->tax != 0) {
            $expense->taxname = [];
            array_push($expense->taxname, $expense->tax_name . '|' . $expense->taxrate);
        }
        if ($expense->tax2 != 0) {
            array_push($expense->taxname, $expense->tax_name2 . '|' . $expense->taxrate2);
        }
        echo json_encode($expense);
    }

    /* Add new invoice or update existing */
    public function invoice_old_select($id = '')
    {
        $series = $this->invoices_model->get_series();
        if (empty($series)) {
            set_alert('warning', _l('no_billling_series_in_account', _l('billing')));
            redirect(admin_url('billing/config/series'));
        }

        if ($this->input->post()) {
            $invoice_data = $this->input->post();
            if ($id == '') {
                if (!has_permission('invoices', '', 'create')) {
                    access_denied('invoices');
                }
                $id = $this->invoices_model->add($invoice_data);
                if ($id) {
                    set_alert('success', _l('added_successfully', _l('invoice')));
                    $redUrl = admin_url('invoices/list_invoices/' . $id);

                    if (isset($invoice_data['save_and_record_payment'])) {
                        $this->session->set_userdata('record_payment', true);
                    } elseif (isset($invoice_data['save_and_send_later'])) {
                        $this->session->set_userdata('send_later', true);
                    }

                    redirect($redUrl);
                }
            } else {
                if (!has_permission('invoices', '', 'edit')) {
                    access_denied('invoices');
                }
                $success = $this->invoices_model->update($invoice_data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('invoice')));
                }
                redirect(admin_url('invoices/list_invoices/' . $id));
            }
        }
        if ($id == '') {
            $title = _l('create_new_invoice');
            $data['billable_tasks'] = [];
        } else {
            $invoice = $this->invoices_model->get($id);

            if (!$invoice || !user_can_view_invoice($id)) {
                blank_page(_l('invoice_not_found'));
            }

            $data['invoices_to_merge'] = $this->invoices_model->check_for_merge_invoice($invoice->clientid, $invoice->id);
            $data['expenses_to_bill'] = $this->invoices_model->get_expenses_to_bill($invoice->clientid);

            $data['invoice'] = $invoice;
            $data['edit'] = true;
            $data['billable_tasks'] = $this->tasks_model->get_billable_tasks($invoice->clientid, !empty($invoice->project_id) ? $invoice->project_id : '');

            $title = _l('edit', _l('invoice_lowercase')) . ' - ' . format_invoice_number($invoice->id);
        }

        if ($this->input->get('customer_id')) {
            $data['customer_id'] = $this->input->get('customer_id');
        }

        $this->load->model('payment_modes_model');
        $data['payment_modes'] = $this->payment_modes_model->get('', [
            'expenses_only !=' => 1,
        ]);

        $this->load->model('taxes_model');
        $data['taxes'] = $this->taxes_model->get();
        $this->load->model('invoice_items_model');

        $data['ajaxItems'] = false;
        if (total_rows(db_prefix() . 'items') <= ajax_on_total_items()) {
            $data['items'] = $this->invoice_items_model->get_grouped();

            if (!empty($data['items'])) {
                foreach ($data['items'] as $key => $group) {
                    foreach ($group as $itemkey => $item) {
                        if (isset($data['items'][$key][$itemkey]['group_type'])) {
                            if ($data['items'][$key][$itemkey]['group_type'] == '1') {
                                $data['items'][$key][$itemkey]['name'] = $item['group_name'] . ' (Products)';
                                $data['items'][$key][$itemkey]['group_name'] = $item['group_name'] . ' (Products)';

                            } else if ($data['items'][$key][$itemkey]['group_type'] == '2') {
                                $data['items'][$key][$itemkey]['name'] = $item['group_name'] . ' (Services)';
                                $data['items'][$key][$itemkey]['group_name'] = $item['group_name'] . ' (Services)';

                            }
                        }
                    }
                }
            }

        } else {
            $data['items'] = [];
            $data['ajaxItems'] = true;
        }
        //echo '<pre>'; var_dump($data['items']); exit;
        $data['items_groups'] = $this->invoice_items_model->get_groups();
        //echo '<pre>'; var_dump($data['items_groups']); exit;
        if (!empty($data['items_groups'])) {
            foreach ($data['items_groups'] as $key => $group) {
                if ($group['group_type'] == '1') {
                    $data['items_groups'][$key]['name'] = $group['name'] . ' (Products)';
                } else if ($group['group_type'] == '2') {
                    $data['items_groups'][$key]['name'] = $group['name'] . ' (Services)';
                }
            }
        }

        $this->load->model('currencies_model');
        $data['currencies'] = $this->currencies_model->get();

        $data['base_currency'] = $this->currencies_model->get_base_currency();

        $data['staff'] = $this->staff_model->get('', ['active' => 1]);
        $data['title'] = $title;
        $data['bodyclass'] = 'invoice';

        $settings['vat_payer'] = (int)get_option('settings_company_vat_payer');
        $settings['vat_on_collection'] = (int)get_option('settings_company_vat_on_collection');
        if ($settings['vat_on_collection'] == 1) {
            $display_vat_collection = '<p>' . _l('invoice_notice_vat_on_collection') . '</p>';
            $settings['display_vat_collection'] = $display_vat_collection;
        }
        $settings['show_tax_per_item'] = (int)get_option('show_tax_per_item');

        $data['settings'] = $settings;

        $data['invoice_settings'] = get_option('invoice_settings');
        $data['invoice_settings'] = json_decode($data['invoice_settings'], true);
        $this->load->view('admin/invoices/invoice', $data);
    }

    public function invoice($id = '')
    {
        $data['active_menu'] = 'invoices';
        $series = $this->invoices_model->get_series();
        if (empty($series)) {
            set_alert('warning', _l('no_billling_series_in_account', _l('billing')));
            redirect(admin_url('billing/config/series'));
        }

        if ($this->input->post()) {
            $invoice_data = $this->input->post();
            if ($id == '') {
                if (!has_permission('invoices', '', 'create')) {
                    access_denied('invoices');
                }
                $id = $this->invoices_model->add($invoice_data);
                if ($id) {
                    set_alert('success', _l('added_successfully', _l('invoice')));
                    $redUrl = admin_url('invoices/list_invoices/' . $id);

                    if (isset($invoice_data['save_and_record_payment'])) {
                        $this->session->set_userdata('record_payment', true);
                    } elseif (isset($invoice_data['save_and_send_later'])) {
                        $this->session->set_userdata('send_later', true);
                    }

                    redirect($redUrl);
                }
            } else {
                if (!has_permission('invoices', '', 'edit')) {
                    access_denied('invoices');
                }
                $success = $this->invoices_model->update($invoice_data, $id);
                if ($success) {
                    set_alert('success', _l('updated_successfully', _l('invoice')));
                }
                redirect(admin_url('invoices/list_invoices/' . $id));
            }
        }
        if ($id == '') {
            $title = _l('create_new_invoice');
            $data['billable_tasks'] = [];
        } else {
            $invoice = $this->invoices_model->get($id);

            if (!$invoice || !user_can_view_invoice($id)) {
                blank_page(_l('invoice_not_found'));
            }

            $data['invoices_to_merge'] = $this->invoices_model->check_for_merge_invoice($invoice->clientid, $invoice->id);
            $data['expenses_to_bill'] = $this->invoices_model->get_expenses_to_bill($invoice->clientid);

            $data['invoice'] = $invoice;
            $data['edit'] = true;
            $data['billable_tasks'] = $this->tasks_model->get_billable_tasks($invoice->clientid, !empty($invoice->project_id) ? $invoice->project_id : '');

            $title = _l('edit', _l('invoice_lowercase')) . ' - ' . format_invoice_number($invoice->id);
        }

        if ($this->input->get('customer_id')) {
            $data['customer_id'] = $this->input->get('customer_id');
        }

        $this->load->model('payment_modes_model');
        $data['payment_modes'] = $this->payment_modes_model->get('', [
            'expenses_only !=' => 1,
        ]);

        $this->load->model('taxes_model');
        $data['taxes'] = $this->taxes_model->get();
        $this->load->model('invoice_items_model');

        $data['ajaxItems'] = false;
        if (total_rows(db_prefix() . 'items') <= ajax_on_total_items()) {
            $data['items'] = $this->invoice_items_model->get_grouped();

            if (!empty($data['items'])) {
                foreach ($data['items'] as $key => $group) {
                    foreach ($group as $itemkey => $item) {
                        if (isset($data['items'][$key][$itemkey]['group_type'])) {
                            if ($data['items'][$key][$itemkey]['group_type'] == '1') {
                                $data['items'][$key][$itemkey]['name'] = $item['group_name'] . ' (Products)';
                                $data['items'][$key][$itemkey]['group_name'] = $item['group_name'] . ' (Products)';

                            } else if ($data['items'][$key][$itemkey]['group_type'] == '2') {
                                $data['items'][$key][$itemkey]['name'] = $item['group_name'] . ' (Services)';
                                $data['items'][$key][$itemkey]['group_name'] = $item['group_name'] . ' (Services)';

                            }
                        }
                    }
                }
            }

        } else {
            $data['items'] = [];
            $data['ajaxItems'] = true;
        }
        //echo '<pre>'; var_dump($data['items']); exit;
        $data['items_groups'] = $this->invoice_items_model->get_groups();
        //echo '<pre>'; var_dump($data['items_groups']); exit;
        if (!empty($data['items_groups'])) {
            foreach ($data['items_groups'] as $key => $group) {
                if ($group['group_type'] == '1') {
                    $data['items_groups'][$key]['name'] = $group['name'] . ' (Products)';
                } else if ($group['group_type'] == '2') {
                    $data['items_groups'][$key]['name'] = $group['name'] . ' (Services)';
                }
            }
        }

        $this->load->model('currencies_model');
        $data['currencies'] = $this->currencies_model->get();

        $data['base_currency'] = $this->currencies_model->get_base_currency();

        $data['staff'] = $this->staff_model->get('', ['active' => 1]);
        $data['title'] = $title;
        $data['bodyclass'] = 'invoice';

        $settings['vat_payer'] = (int)get_option('settings_company_vat_payer');
        $settings['vat_on_collection'] = (int)get_option('settings_company_vat_on_collection');
        if ($settings['vat_on_collection'] == 1) {
            $display_vat_collection = '<p>' . _l('invoice_notice_vat_on_collection') . '</p>';
            $settings['display_vat_collection'] = $display_vat_collection;
        }
        $settings['show_tax_per_item'] = (int)get_option('show_tax_per_item');

        $data['settings'] = $settings;

        $data['invoice_settings'] = get_option('invoice_settings');
        $data['invoice_settings'] = json_decode($data['invoice_settings'], true);
        $this->load->view('admin/invoices/invoice2', $data);
    }

    /* Get all invoice data used when user click on invoiec number in a datatable left side*/
    public function get_invoice_data_ajax($id)
    {
        if (!has_permission('invoices', '', 'view')
            && !has_permission('invoices', '', 'view_own')
            && get_option('allow_staff_view_invoices_assigned') == '0') {
            echo _l('access_denied');
            die;
        }

        if (!$id) {
            die(_l('invoice_not_found'));
        }

        $invoice = $this->invoices_model->get($id);


        if (!$invoice || !user_can_view_invoice($id)) {
            echo _l('invoice_not_found');
            die;
        }

        $template_name = 'invoice_send_to_customer';

        if ($invoice->sent == 1) {
            $template_name = 'invoice_send_to_customer_already_sent';
        }

        $data = prepare_mail_preview_data($template_name, $invoice->clientid);

        // Check for recorded payments
        $this->load->model('payments_model');
        $data['invoices_to_merge'] = $this->invoices_model->check_for_merge_invoice($invoice->clientid, $id);
        $data['members'] = $this->staff_model->get('', ['active' => 1]);
        $data['payments'] = $this->payments_model->get_invoice_payments($id);
        $data['activity'] = $this->invoices_model->get_invoice_activity($id);
        $data['totalNotes'] = total_rows(db_prefix() . 'notes', ['rel_id' => $id, 'rel_type' => 'invoice']);
        $data['invoice_recurring_invoices'] = $this->invoices_model->get_invoice_recurring_invoices($id);

        $data['applied_credits'] = $this->credit_notes_model->get_applied_invoice_credits($id);
        // This data is used only when credit can be applied to invoice
        if (credits_can_be_applied_to_invoice($invoice->status)) {
            $data['credits_available'] = $this->credit_notes_model->total_remaining_credits_by_customer($invoice->clientid);

            if ($data['credits_available'] > 0) {
                $data['open_credits'] = $this->credit_notes_model->get_open_credits($invoice->clientid);
            }

            $customer_currency = $this->clients_model->get_customer_default_currency($invoice->clientid);
            $this->load->model('currencies_model');

            if ($customer_currency != 0) {
                $data['customer_currency'] = $this->currencies_model->get($customer_currency);
            } else {
                $data['customer_currency'] = $this->currencies_model->get_base_currency();
            }
        }

        $data['invoice'] = $invoice;
        $data['invoice']->options = json_decode(get_option('invoice_settings'), TRUE);
        //Custom TMG - EDIConnect
        if (!empty($data['invoice']->items)) {
            foreach ($data['invoice']->items as $k => $item_inv) {
                $replaced_value = '';
                $has_tag = false;
                $inv_date = $data['invoice']->date;

                if (strpos($item_inv['description'], '[SAPTAMANA]') !== false) {
                    $replaced_value = _l('weeknumber') . ' ' . date("W", strtotime($inv_date));

                    $data['invoice']->items[$k]['description'] = str_replace('[SAPTAMANA]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[SAPTAMANA TRECUTA]') !== false) {
                    $replaced_value = _l('weeknumber') . ' ' . date('W', strtotime('-1 week', strtotime($inv_date)));

                    $data['invoice']->items[$k]['description'] = str_replace('[SAPTAMANA TRECUTA]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[SAPTAMANA URMATOARE]') !== false) {
                    $replaced_value = _l('weeknumber') . ' ' . date('W', strtotime('+1 week', strtotime($inv_date)));

                    $data['invoice']->items[$k]['description'] = str_replace('[SAPTAMANA URMATOARE]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[LUNA]') !== false) {
                    $replaced_value = ' ' . date("M Y", strtotime($inv_date));

                    $data['invoice']->items[$k]['description'] = str_replace('[LUNA]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[LUNA VIITOARE]') !== false) {
                    $replaced_value = date("M Y", mktime(0, 0, 0, date("m", strtotime($inv_date)) + 1, 1, date("Y", strtotime($inv_date))));

                    $data['invoice']->items[$k]['description'] = str_replace('[LUNA VIITOARE]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[LUNA TRECUTA]') !== false) {
                    $replaced_value = date("M Y", mktime(0, 0, 0, date("m", strtotime($inv_date)) - 1, 1, date("Y", strtotime($inv_date))));

                    $data['invoice']->items[$k]['description'] = str_replace('[LUNA TRECUTA]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[AN]') !== false) {
                    $replaced_value = ' ' . date("Y", strtotime($inv_date));

                    $data['invoice']->items[$k]['description'] = str_replace('[AN]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[ANUL TRECUT]') !== false) {
                    $replaced_value = date('Y', strtotime('-1 year', strtotime($inv_date)));

                    $data['invoice']->items[$k]['description'] = str_replace('[ANUL TRECUT]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                } else if (strpos($item_inv['description'], '[ANUL URMATOR]') !== false) {
                    $replaced_value = date('Y', strtotime('+1 year', strtotime($inv_date)));

                    $data['invoice']->items[$k]['description'] = str_replace('[ANUL URMATOR]', $replaced_value, $item_inv['description']);
                    $has_tag = true;
                }


                if ($has_tag) {
                    $data['invoice']->items[$k]['description'] .= '<span class="badge badge-primary" style="margin-left:6px;">Dinamic</span>';
                }
            }
        }
        $data['record_payment'] = false;
        $data['send_later'] = false;

        if ($this->session->has_userdata('record_payment')) {
            $data['record_payment'] = true;
            $this->session->unset_userdata('record_payment');
        } elseif ($this->session->has_userdata('send_later')) {
            $data['send_later'] = true;
            $this->session->unset_userdata('send_later');
        }

        $this->load->helper('billing/efactura_invoice');
        $data['invoiceBilder'] = new EfacturaInvoiceBuilder();

        $this->load->view('admin/invoices/invoice_preview_template', $data);
    }

    public function apply_credits($invoice_id)
    {
        $total_credits_applied = 0;
        foreach ($this->input->post('amount') as $credit_id => $amount) {
            $success = $this->credit_notes_model->apply_credits($credit_id, [
                'invoice_id' => $invoice_id,
                'amount' => $amount,
            ]);
            if ($success) {
                $total_credits_applied++;
            }
        }

        if ($total_credits_applied > 0) {
            update_invoice_status($invoice_id, true);
            set_alert('success', _l('invoice_credits_applied'));
        }
        redirect(admin_url('invoices/list_invoices/' . $invoice_id));
    }

    public function get_invoices_total()
    {
        if ($this->input->post()) {
            load_invoices_total_template();
        }
    }

    /* Record new inoice payment view */
    public function record_invoice_payment_ajax($id)
    {
        $this->load->model('payment_modes_model');
        $this->load->model('payments_model');
        $data['payment_modes'] = $this->payment_modes_model->get('', [
            'expenses_only !=' => 1,
        ]);
        $data['invoice'] = $this->invoices_model->get($id);
        $data['payments'] = $this->payments_model->get_invoice_payments($id);
        $this->load->view('admin/invoices/record_payment_template', $data);
    }

    /* This is where invoice payment record $_POST data is send */
    public function record_payment()
    {
        if (!has_permission('payments', '', 'create')) {
            access_denied('Record Payment');
        }
        if ($this->input->post()) {
            $this->load->model('payments_model');
            $id = $this->payments_model->process_payment($this->input->post(), '');
            if ($id) {
                set_alert('success', _l('invoice_payment_recorded'));
                redirect(admin_url('payments/payment/' . $id));
            } else {
                set_alert('danger', _l('invoice_payment_record_failed'));
            }
            redirect(admin_url('invoices/list_invoices/' . $this->input->post('invoiceid')));
        }
    }

    /* Send invoice to email */
    public function send_to_email($id)
    {
        $canView = user_can_view_invoice($id);
        if (!$canView) {
            access_denied('Invoices');
        } else {
            if (!has_permission('invoices', '', 'view') && !has_permission('invoices', '', 'view_own') && $canView == false) {
                access_denied('Invoices');
            }
        }

        try {
            $statementData = [];
            if ($this->input->post('attach_statement')) {
                $statementData['attach'] = true;
                $statementData['from'] = to_sql_date($this->input->post('statement_from'));
                $statementData['to'] = to_sql_date($this->input->post('statement_to'));
            }

            $success = $this->invoices_model->send_invoice_to_client(
                $id,
                '',
                $this->input->post('attach_pdf'),
                $this->input->post('cc'),
                false,
                $statementData
            );
        } catch (Exception $e) {
            $message = $e->getMessage();
            echo $message;
            if (strpos($message, 'Unable to get the size of the image') !== false) {
                show_pdf_unable_to_get_image_size_error();
            }
            die;
        }

        // In case client use another language
        load_admin_language();
        if ($success) {
            set_alert('success', _l('invoice_sent_to_client_success'));
        } else {
            set_alert('danger', _l('invoice_sent_to_client_fail'));
        }
        redirect(admin_url('invoices/list_invoices/' . $id));
    }

    /* Delete invoice payment*/
    public function delete_payment($id, $invoiceid)
    {
        if (!has_permission('payments', '', 'delete')) {
            access_denied('payments');
        }
        $this->load->model('payments_model');
        if (!$id) {
            redirect(admin_url('payments'));
        }
        $response = $this->payments_model->delete($id);
        if ($response == true) {
            set_alert('success', _l('deleted', _l('payment')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('payment_lowercase')));
        }
        redirect(admin_url('invoices/list_invoices/' . $invoiceid));
    }

    /* Delete invoice */
    public function delete($id)
    {
        if (!has_permission('invoices', '', 'delete')) {
            access_denied('invoices');
        }
        if (!$id) {
            redirect(admin_url('invoices/list_invoices'));
        }
        $success = $this->invoices_model->delete($id);

        if ($success) {
            set_alert('success', _l('deleted', _l('invoice')));
        } else {
            set_alert('warning', _l('problem_deleting', _l('invoice_lowercase')));
        }
        if (strpos($_SERVER['HTTP_REFERER'], 'list_invoices') !== false) {
            redirect(admin_url('invoices/list_invoices'));
        } else {
            redirect($_SERVER['HTTP_REFERER']);
        }
    }

    public function delete_attachment($id)
    {
        $file = $this->misc_model->get_file($id);
        if ($file->staffid == get_staff_user_id() || is_admin()) {
            echo $this->invoices_model->delete_attachment($id);
        } else {
            header('HTTP/1.0 400 Bad error');
            echo _l('access_denied');
            die;
        }
    }

    /* Will send overdue notice to client */
    public function send_overdue_notice($id)
    {
        $canView = user_can_view_invoice($id);
        if (!$canView) {
            access_denied('Invoices');
        } else {
            if (!has_permission('invoices', '', 'view') && !has_permission('invoices', '', 'view_own') && $canView == false) {
                access_denied('Invoices');
            }
        }

        $send = $this->invoices_model->send_invoice_overdue_notice($id);
        if ($send) {
            set_alert('success', _l('invoice_overdue_reminder_sent'));
        } else {
            set_alert('warning', _l('invoice_reminder_send_problem'));
        }
        redirect(admin_url('invoices/list_invoices/' . $id));
    }

    /* Generates invoice PDF and senting to email of $send_to_email = true is passed */
    public function pdf($id)
    {
        if (!$id) {
            redirect(admin_url('invoices/list_invoices'));
        }

        $canView = user_can_view_invoice($id);
        if (!$canView) {
            access_denied('Invoices');
        } else {
            if (!has_permission('invoices', '', 'view') && !has_permission('invoices', '', 'view_own') && $canView == false) {
                access_denied('Invoices');
            }
        }

        $invoice = $this->invoices_model->get($id);
        $invoice = hooks()->apply_filters('before_admin_view_invoice_pdf', $invoice);
        $invoice_number = format_invoice_number($invoice->id);

        /*
        $_is_draft = (isset($invoice) && $invoice->status == Invoices_model::STATUS_DRAFT) ? true : false;
        $_invoice_number = str_pad($__number, get_option('number_padding_prefixes'), '0', STR_PAD_LEFT);
        $isedit = isset($invoice) ? 'true' : 'false';
        $data_original_number = isset($invoice) ? $invoice->number : 'false';
        */

        $type = 'D';

        if ($this->input->get('output_type')) {
            $type = $this->input->get('output_type');
        }

        if ($this->input->get('print')) {
            $type = 'I';
        }

        $company = [
            'email' => get_option('invoice_company_email'),
            'name' => get_option('invoice_company_name'),
            'vat' => get_option('company_vat'),
            'vat_eu' => get_option('company_vat_eu'),
            'trade_no' => get_option('company_trade'),
            'address' => get_option('invoice_company_address'),
            'city' => get_option('invoice_company_city'),
            'state' => get_option('company_state'),
            'postal_code' => get_option('invoice_company_postal_code'),
            'country_code' => get_option('invoice_company_country_code'),
            'website' => get_option('invoice_company_website'),
            'phonenumber' => get_option('invoice_company_phonenumber'),
            'fax' => get_option('invoice_company_fax'),

        ];


        $data = [
            'invoice_number' => $invoice_number,
            'return_type' => 'pdf',
            'output' => $type,
            'company' => $company,
            'invoice' => $invoice, //json_decode(json_encode($invoice),true),
        ];


        $settings['vat_payer'] = (int)get_option('settings_company_vat_payer');
        $settings['vat_on_collection'] = (int)get_option('settings_company_vat_on_collection');
        if ($settings['vat_on_collection'] == 1) {
            $display_vat_collection = '<p>' . _l('invoice_notice_vat_on_collection') . '</p>';
            $settings['display_vat_collection'] = $display_vat_collection;
        }
        $settings['show_tax_per_item'] = (int)get_option('show_tax_per_item');

        $data['settings'] = $settings;

        $this->load->model('billing/billing_bank_model');
        $banks = $this->billing_bank_model->get_active();

        $data['banks'] = $banks;

        // $this->_template_invoice($data);


        $this->load->library('billing/crm_invoice', $invoice);
        $invoice = $this->crm_invoice->prepare_document();
        $invoice->company = $company;
        $invoice->settings = $settings;

        if ($invoice->template == '') {
            /*Original*/
            try {
                $pdf = invoice_pdf($invoice);
            } catch (Exception $e) {
                $message = $e->getMessage();
                echo $message;
                if (strpos($message, 'Unable to get the size of the image') !== false) {
                    show_pdf_unable_to_get_image_size_error();
                }
                die;
            }
            $pdf->Output(mb_strtoupper(slug_it($invoice_number)) . '.pdf', $type);


        } else {


        }


    }

    public function _template_invoice($data)
    {
        $this->load->library('mpdf_lib');

        $defaultConfig = (new Mpdf\Config\ConfigVariables())->getDefaults();
        $fontDirs = $defaultConfig['fontDir'];

        $defaultFontConfig = (new Mpdf\Config\FontVariables())->getDefaults();
        $fontData = $defaultFontConfig['fontdata'];

        $mpdf = new MPDF\MPDF([
            'mode' => 'utf-8',
            'format' => 'A4-P',
            'default_font_size' => NULL,
            'default_font' => NULL,
            'mgl' => 0,
            'fontDir' => array_merge($fontDirs, [
                $_SERVER["DOCUMENT_ROOT"] . '/assets/invoice-design-v2/fonts/poppins',
            ]),
            'fontdata' => $fontData + [
                    'Poppins' => [
                        'R' => 'Poppins-Regular.ttf',
                        'I' => 'Poppins-Italic.ttf',
                        'B' => 'Poppins-Bold.ttf',
                    ],
                ],
        ]);

        $mpdf->SetDisplayMode('fullpage');

        //		$mpdf = new MPDF\MPDF([
        //			'mode' => 'utf-8',
        //			'format' => 'A4-P',
        //			'default_font_size' => NULL,
        //			'default_font' => NULL,
        //			'mgl' => 0,
        //		]);
        $filename = mb_strtoupper(slug_it($data['invoice_number'])) . '.pdf';
        $this->load->library('billing/crm_invoice', $data['invoice']);

        //  pre($data);
        $data['invoice'] = $this->crm_invoice->prepare_document();
        if ($data['invoice']->template == '' or $data['invoice']->template == null) {
            $html = $this->load->view('billing/templates/v2/main', $data, true);//TODO MIA
        } else {
            $html = $this->load->view('billing/templates/' . $data['invoice']->template . '/main', $data, true);
        }
        //echo '<pre>';
        //print_r($data);
        //echo '</pre>';
        //die;

        if (isset($_GET['output_type'])) {
            if ($_GET['output_type'] == 'HTML') {
                echo '<style>
            body{width: 793.7007874px; margin: 0px auto; background: white;}
            html{width: 100%;background: #c3c2c221;}
            </style>';
                echo '<div class="pgborder" style=" width: 100%; border: 1px solid #000000f5; position: relative; top: 1122.519685px; z-index: 100; "></div>';
                echo '<div class="pgborder" style=" width: 100%; border: 1px solid #000000f5; position: relative; top: 1587.4015748px; z-index: 100; "></div>';
                echo '<div class="pgborder" style=" width: 100%; border: 1px solid #000000f5; position: relative; top: 2381.1023622px; z-index: 100; "></div>';

                echo $html;
                die();
            }
        }


        $mpdf->setAutoTopMargin = 'stretch';
        $mpdf->setAutoBottomMargin = 'stretch';
        $mpdf->WriteHTML($html);

        $mpdf->Output($filename, $data['output']);
    }

    public function mark_as_sent($id)
    {
        if (!$id) {
            redirect(admin_url('invoices/list_invoices'));
        }
        if (!user_can_view_invoice($id)) {
            access_denied('Invoice Mark As Sent');
        }

        $success = $this->invoices_model->set_invoice_sent($id, true);

        if ($success) {
            set_alert('success', _l('invoice_marked_as_sent'));
        } else {
            set_alert('warning', _l('invoice_marked_as_sent_failed'));
        }

        redirect(admin_url('invoices/list_invoices/' . $id));
    }

    public function get_due_date()
    {
        if ($this->input->post()) {
            $date = $this->input->post('date');
            $duedate = '';
            if (get_option('invoice_due_after') != 0) {
                $date = to_sql_date($date);
                $d = date('Y-m-d', strtotime('+' . get_option('invoice_due_after') . ' DAY', strtotime($date)));
                $duedate = _d($d);
                echo $duedate;
            }
        }
    }

    private function excelToJson($filePath)
    {
        $excelReader = new SpreadsheetReader($filePath);

        $headers = [];
        $data = [];

        // Get the headers from the first row
        foreach ($excelReader as $row) {
            $headers = $row;
            break;
        }

        // Process the remaining rows and exclude the row with column names
        $isFirstRow = true; // Flag to skip the first row
        foreach ($excelReader as $row) {
            // Check if the row is the first row with column names
            if ($isFirstRow) {
                $isFirstRow = false;
                continue;
            }

            // Check if the first column has data
            if (!empty($row[0])) {
                $rowData = [];
                // Loop through each cell in the row
                foreach ($row as $index => $cell) {
                    // Use the header as the key for the associative array
                    $header = isset($headers[$index]) ? $headers[$index] : 'unknown';
                    $rowData[$header] = $cell;
                }
                // Add the row data to the main data array
                $data[] = $rowData;
            }
        }

        return $data;
    }

    private function extractNamePart($type, $string)
    {
        if (empty($string)) {
            return '';
        }

        // Split the string into an array of words
        $words = explode(' ', $string);

        // Choose the part based on the type
        if ($type === 'firstname') {
            // Return the first word (first name)
            return isset($words[0]) ? $words[0] : '';
        } elseif ($type === 'lastname') {
            // Return the second word onwards as the last name
            return isset($words[1]) ? implode(' ', array_slice($words, 1)) : '';
        } else {
            // Invalid type, return an empty string
            return '';
        }
    }

    /**
     * Import helpers
     * Helpers to create directories and files needed for import
     * same we hav find in migration 116
     * sometimes the migration is not applied
     * until then we do this check
     */
    public function ensure_invoice_folders_exist()
    {
        $this->create_directory(INVOICES_IMPORT_FOLDER);
        $this->create_directory(INVOICES_IMPORT_FOLDER_IMPORTS);
        $this->create_directory(INVOICES_IMPORT_FOLDER_DATA);
    }

    private function create_directory($path)
    {
        if (!is_dir($path)) {
            mkdir($path, 0755, true); // Recursive directory creation
            $this->create_htaccess($path);
            $this->create_empty_json_file($path);
        }
    }

    private function create_htaccess($path)
    {
        $htaccessPath = $path . '.htaccess';
        if (!file_exists($htaccessPath)) {
            $fp = fopen($htaccessPath, 'w');
            if ($fp) {
                fwrite($fp, 'Order Deny,Allow' . PHP_EOL . 'Deny from all');
                fclose($fp);
            }
        }
    }

    private function create_empty_json_file($path)
    {
        $jsonFilePath = $path . 'imports.json';
        if (!file_exists($jsonFilePath)) {
            file_put_contents($jsonFilePath, json_encode([]));
        }
    }

}
