<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Add option
 *
 * @param string $name Option name (required|unique)
 * @param string $value Option value
 * @param integer $autoload Whether to autoload this option
 *
 * @since  Version 1.0.1
 *
 */
function add_option($name, $value = '', $autoload = 1)
{
    if (!option_exists($name)) {
        $CI = &get_instance();

        $newData = [
            'name' => $name,
            'value' => $value,
        ];

        if ($CI->db->field_exists('autoload', db_prefix() . 'options')) {
            $newData['autoload'] = $autoload;
        }

        $CI->db->insert(db_prefix() . 'options', $newData);

        $insert_id = $CI->db->insert_id();

        if ($insert_id) {
            return true;
        }

        return false;
    }

    return false;
}

/**
 * Get option value
 * @param string $name Option name
 * @return mixed
 */
function get_option($name)
{
    $CI = &get_instance();
    if (!class_exists('app', false)) {
        $CI->load->library('app');
    }

    return $CI->app->get_option($name);
}

/**
 * Updates option by name
 *
 * @param string $name Option name
 * @param string $value Option Value
 * @param mixed $autoload Whether to update the autoload
 *
 * @return boolean
 */
function update_option($name, $value, $autoload = null)
{
    /**
     * Create the option if not exists
     * @since  2.3.3
     */
    if (!option_exists($name)) {
        return add_option($name, $value, $autoload === null ? 1 : 0);
    }

    $CI = &get_instance();

    $CI->db->where('name', $name);
    $data = ['value' => $value];

    if ($autoload) {
        $data['autoload'] = $autoload;
    }

    $CI->db->update(db_prefix() . 'options', $data);

    if ($CI->db->affected_rows() > 0) {
        return true;
    }

    return false;
}

/**
 * Delete option
 * @param mixed $name option name
 * @return boolean
 * @since  Version 1.0.4
 */
function delete_option($name)
{
    $CI = &get_instance();
    $CI->db->where('name', $name);
    $CI->db->delete(db_prefix() . 'options');

    return (bool)$CI->db->affected_rows();
}

/**
 * @param string $name option name
 *
 * @return boolean
 * @since  2.3.3
 * Check whether an option exists
 *
 */
function option_exists($name)
{
    return total_rows(db_prefix() . 'options', [
            'name' => $name,
        ]) > 0;
}

function app_init_settings_tabs()
{
    $CI = &get_instance();

    /*$CI->app_tabs->add_settings_tab('logos', [
        'name'     => _l('Logo-uri'),
        'view'     => 'admin/settings/includes/logos',
        'position' => 10,
    ]);*/

    $CI->app_tabs->add_settings_tab('general', [
        'name' => _l('settings_group_general'),
        'view' => 'admin/settings/includes/general',
        'position' => 5,
    ]);

    $CI->app_tabs->add_settings_tab('company', [
        'name' => _l('company_information'),
        'view' => 'admin/settings/includes/company',
        'position' => 10,
    ]);

    if ($CI->app_modules->is_active('dashboards')) {
        $CI->app_tabs->add_settings_tab('dashboards', [
            'name' => _l('dashboards_settings'),
            'view' => 'admin/settings/includes/dashboard',
            'position' => 10,
        ]);
    }
}
