<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * @param array $data additional data passed from view role.php and member.php
 * @return array
 * @since  2.3.3
 * Get available staff permissions, modules can use the filter too to hook permissions
 */
function get_available_staff_permissions($data = [])
{
    $viewGlobalName = _l('permission_view') . '(' . _l('permission_global') . ')';

    $allPermissionsArray = [
        'view_own' => _l('permission_view_own'),
        'view' => $viewGlobalName,
        'create' => _l('permission_create'),
        'edit' => _l('permission_edit'),
        'delete' => _l('permission_delete'),
    ];

    $withoutViewOwnPermissionsArray = [
        'view' => $viewGlobalName,
        'create' => _l('permission_create'),
        'edit' => _l('permission_edit'),
        'delete' => _l('permission_delete'),
    ];

    $withNotApplicableViewOwn = array_merge(['view_own' => ['not_applicable' => true, 'name' => _l('permission_view_own')]], $withoutViewOwnPermissionsArray);


    $corePermissions = [
        'bulk_pdf_exporter' => [
            'name' => _l('bulk_pdf_exporter'),
            'capabilities' => [
                'view' => $viewGlobalName,
            ],
        ],
        'contracts' => [
            'name' => _l('contracts'),
            'capabilities' => $allPermissionsArray,
            'category' => _l('legal_permissions')
        ],
        'credit_notes' => [
            'name' => _l('credit_notes'),
            'capabilities' => $allPermissionsArray,
            'category' => _l('finance_permissions')
        ],
        'customers' => [
            'name' => _l('clients'),
            'capabilities' => $withNotApplicableViewOwn,
            'help' => [
                'view_own' => _l('sales_permissions'),
            ],
            'category' => _l('sales_permissions'),
        ],
        'email_templates' => [
            'name' => _l('email_templates'),
            'capabilities' => [
                'view' => $viewGlobalName,
                'edit' => _l('permission_edit'),
            ],
            'category' => _l('marketing_permissions')
        ],
        'estimates' => [
            'name' => _l('estimates'),
            'capabilities' => $allPermissionsArray,
            'category' => _l('sales_permissions')
        ],
        'expenses' => [
            'name' => _l('expenses'),
            'capabilities' => $allPermissionsArray,
            'category' => _l('finance_permissions')
        ],
        'invoices' => [
            'name' => _l('invoices'),
            'capabilities' => $allPermissionsArray,
            'category' => _l('finance_permissions')
        ],
        /*  'monitoring' => [
              'name'         => _l('monitoring'),
              'capabilities' => [
                  'view' => $viewGlobalName,
              ],
          ],*/
        'items' => [
            'name' => _l('items'),
            'capabilities' => $withoutViewOwnPermissionsArray,
            'category' => _l('sales_permissions'),
        ],
        'knowledge_base' => [
            'name' => _l('knowledge_base'),
            'capabilities' => $withoutViewOwnPermissionsArray,
            'category' => _l('helpdesk_permissions')
        ],
        'payments' => [
            'name' => _l('payments'),
            'capabilities' => $withNotApplicableViewOwn,
            'help' => [
                'view_own' => _l('permission_payments_based_on_invoices'),
            ],
            'category' => _l('finance_permissions')
        ],
        'projects' => [
            'name' => _l('projects'),
            'capabilities' => $withNotApplicableViewOwn,
            'help' => [
                'view' => _l('help_project_permissions'),
                'view_own' => _l('permission_projects_based_on_assignee'),
            ],
            'category' => _l('project_tasks_permissions')
        ],
        'proposals' => [
            'name' => _l('proposals'),
            'capabilities' => $allPermissionsArray,
            'category' => _l('sales_permissions')
        ],
        'reports' => [
            'name' => _l('reports'),
            'capabilities' => [
                'view' => $viewGlobalName,
            ],
            'category' => _l('management_permissions'),
        ],
        'roles' => [
            'name' => _l('roles'),
            'capabilities' => $withoutViewOwnPermissionsArray,
            'category' => _l('hr_permissions')
        ],
        'settings' => [
            'name' => _l('settings'),
            'capabilities' => [
                'view' => $viewGlobalName,
                'edit' => _l('permission_edit'),
            ],
            'category' => _l('settings_permissions')
        ],
        'staff' => [
            'name' => _l('staff'),
            'capabilities' => $withoutViewOwnPermissionsArray,
            'category' => _l('hr_permissions')
        ],
        'subscriptions' => [
            'name' => _l('subscriptions'),
            'capabilities' => $allPermissionsArray,
            'category' => _l('sales_permissions'),
        ],
        'tasks' => [
            'name' => _l('tasks'),
            'capabilities' => $withNotApplicableViewOwn,
            'help' => [
                'view' => _l('help_tasks_permissions'),
                'view_own' => _l('permission_tasks_based_on_assignee'),
            ],
            'category' => _l('project_tasks_permissionss')
        ],
        'checklist_templates' => [
            'name' => _l('checklist_templates'),
            'capabilities' => [
                'create' => _l('permission_create'),
                'delete' => _l('permission_delete'),
            ],
            'category' => _l('project_tasks_permissions')
        ],
        'tickets' => [
            'name' => _l('tickets'),
            'capabilities' => [
                'view' => $viewGlobalName,
            ],
            'category' => _l('helpdesk_permissions')
        ],
        'calendar' => [
            'name' => _l('calendar'),
            'capabilities' => [
                'view' => $viewGlobalName,
            ],
            'category' => _l('calendar_permissions')
        ],
        'utilities_media' => [
            'name' => _l('utilities_media'),
            'capabilities' => [
                'view' => $viewGlobalName,
            ],
        ],
        'announcements' => [
            'name' => _l('announcements'),
            'capabilities' => [
                'view' => $viewGlobalName,
            ],
            'category' => _l('calendar_permissions')
        ],


    ];

    $addLeadsPermission = true;
    if (isset($data['staff_id']) && $data['staff_id']) {
        $is_staff_member = is_staff_member($data['staff_id']);
        if (!$is_staff_member) {
            $addLeadsPermission = false;
        }
    }

    if ($addLeadsPermission) {
        $corePermissions['leads'] = [
            'name' => _l('leads_and_opportunities_permission'),
            'capabilities' => [
                'view' => $viewGlobalName,
                'delete' => _l('permission_delete'),
            ],
            'category' => _l('sales_permissions'),
            'help' => [
                'view' => _l('help_leads_permission_view'),
            ],
        ];
    }
    $all_roles = hooks()->apply_filters('staff_permissions', $corePermissions, $data);
    $all_roles = sortArrayByCategory($all_roles);
    uasort($all_roles, 'customRolesSort');

    return $all_roles;
}

function customRolesSort($a, $b)
{
    $priority = array(
        _l('management_permissions') => 1,
        _l('sales_permissions') => 2,
        _l('supply_distribution_permissions') => 3,
        _l('marketing_permissions') => 4,
        _l('hr_permissions') => 5,
        _l('calendar_permissions') => 6,
        _l('helpdesk_permissions') => 7,
        _l('finance_permissions') => 8,
        _l('project_tasks_permissions') => 9,
        _l('legal_permissions') => 10,
        _l('purchase_permissions') => 11,
        _l('settings_permissions') => 12,
    );
    $priorityA = isset($a['category']) && isset($priority[$a['category']]) ? $priority[$a['category']] : PHP_INT_MAX;
    $priorityB = isset($b['category']) && isset($priority[$b['category']]) ? $priority[$b['category']] : PHP_INT_MAX;
    if ($priorityA == $priorityB) {
        return 0;
    }
    return ($priorityA < $priorityB) ? -1 : 1;
}

function sortArrayByCategory($array)
{
    foreach ($array as $key => $row) {
        $category[$key] = (isset($row['category']) ? $row['category'] : _l('uncategorized'));
        $array[$key]['category'] = (isset($row['category']) ? $row['category'] : _l('uncategorized'));
    }
    array_multisort($category, SORT_ASC, $array);
    return $array;
}


/**
 * Get staff by ID or current logged in staff
 * @param mixed $id staff id
 * @return mixed
 */
function get_staff($id = null)
{
    if (empty($id) && isset($GLOBALS['current_user'])) {
        return $GLOBALS['current_user'];
    }

    // Staff not logged in
    if (empty($id)) {
        return null;
    }

    if (!class_exists('staff_model', false)) {
        get_instance()->load->model('staff_model');
    }

    return get_instance()->staff_model->get($id);
}

function is_super_admin($id = null)
{
    $user = get_staff($id);
    return $user ? (isset($user->is_super_admin) ? $user->is_super_admin : 0) : 0;
}

/**
 * Return staff profile image url
 * @param mixed $staff_id
 * @param string $type
 * @return string
 */
function staff_profile_image_url($staff_id, $type = 'small')
{
    $url = base_url('assets/images/user-placeholder.jpg');

    if ((string)$staff_id === (string)get_staff_user_id() && isset($GLOBALS['current_user'])) {
        $staff = $GLOBALS['current_user'];
    } else {
        $CI = &get_instance();
        $CI->db->select('profile_image')
            ->where('staffid', $staff_id);

        $staff = $CI->db->get(db_prefix() . 'staff')->row();
    }

    if ($staff) {
        if (!empty($staff->profile_image)) {
            $profileImagePath = 'uploads/staff_profile_images/' . $staff_id . '/' . $type . '_' . $staff->profile_image;
            if (file_exists($profileImagePath)) {
                $url = base_url($profileImagePath);
            }
        }
    }

    return $url;
}

/**
 * Staff profile image with href
 * @param boolean $id staff id
 * @param array $classes image classes
 * @param string $type
 * @param array $img_attrs additional <img /> attributes
 * @return string
 */
function staff_profile_image($id, $classes = ['staff-profile-image'], $type = 'small', $img_attrs = [])
{
    $url = base_url('assets/images/user-placeholder.jpg');

    $id = trim($id);

    $_attributes = '';
    foreach ($img_attrs as $key => $val) {
        $_attributes .= $key . '=' . '"' . html_escape($val) . '" ';
    }

    $blankImageFormatted = '<img src="' . $url . '" ' . $_attributes . ' class="' . implode(' ', $classes) . '" />';

    if ((string)$id === (string)get_staff_user_id() && isset($GLOBALS['current_user'])) {
        $result = $GLOBALS['current_user'];
    } else {
        $CI = &get_instance();
        $result = $CI->app_object_cache->get('staff-profile-image-data-' . $id);

        if (!$result) {
            $CI->db->select('profile_image,firstname,lastname');
            $CI->db->where('staffid', $id);
            $result = $CI->db->get(db_prefix() . 'staff')->row();
            $CI->app_object_cache->add('staff-profile-image-data-' . $id, $result);
        }
    }

    if (!$result) {
        return $blankImageFormatted;
    }

    if ($result && $result->profile_image !== null) {
        $profileImagePath = 'uploads/staff_profile_images/' . $id . '/' . $type . '_' . $result->profile_image;
        if (file_exists($profileImagePath)) {
            $profile_image = '<img ' . $_attributes . ' src="' . base_url($profileImagePath) . '" class="' . implode(' ', $classes) . '" />';
        } else {
            return $blankImageFormatted;
        }
    } else {
        $profile_image = '<img src="' . $url . '" ' . $_attributes . ' class="' . implode(' ', $classes) . '" />';
    }

    return $profile_image;
}

/**
 * Get staff full name
 * @param string $userid Optional
 * @return string Firstname and Lastname
 */
function get_staff_full_name($userid = '')
{
    $tmpStaffUserId = get_staff_user_id();
    if ($userid == '' || $userid == $tmpStaffUserId) {
        if (isset($GLOBALS['current_user'])) {
            return $GLOBALS['current_user']->firstname . ' ' . $GLOBALS['current_user']->lastname;
        }
        $userid = $tmpStaffUserId;
    }

    $CI = &get_instance();

    $staff = $CI->app_object_cache->get('staff-full-name-data-' . $userid);

    if (!$staff) {
        $CI->db->where('staffid', $userid);
        $staff = $CI->db->select('firstname,lastname')->from(db_prefix() . 'staff')->get()->row();
        $CI->app_object_cache->add('staff-full-name-data-' . $userid, $staff);
    }

    return html_escape($staff ? $staff->firstname . ' ' . $staff->lastname : '');
}

/**
 * Get staff full name
 * @param string $userid Optional
 * @return string Firstname and Lastname
 */
function get_staff_token($userid = '')
{
    $tmpStaffUserId = get_staff_user_id();
    if ($userid == '' || $userid == $tmpStaffUserId) {
        if (isset($GLOBALS['current_user'])) {
            if (empty($GLOBALS['current_user']->token)) {
                return '';
            }
            return $GLOBALS['current_user']->token;
        }
        $userid = $tmpStaffUserId;
    }

    $CI = &get_instance();

    $staff = $CI->app_object_cache->get('staff-token-data-' . $userid);

    if (!$staff) {
        $CI->db->where('staffid', $userid);
        $staff = $CI->db->select('token')->from(db_prefix() . 'staff')->get()->row();
        $CI->app_object_cache->add('staff-token-data-' . $userid, $staff);
    }

    return html_escape($staff ? $staff->token : '');
}

/**
 * Get staff default language
 * @param mixed $staffid
 * @return mixed
 */
function get_staff_default_language($staffid = '')
{
    if (!is_numeric($staffid)) {
        // checking for current user if is admin
        if (isset($GLOBALS['current_user'])) {
            return $GLOBALS['current_user']->default_language;
        }

        $staffid = get_staff_user_id();
    }
    $CI = &get_instance();
    $CI->db->select('default_language');
    $CI->db->from(db_prefix() . 'staff');
    $CI->db->where('staffid', $staffid);
    $staff = $CI->db->get()->row();
    if ($staff) {
        return $staff->default_language;
    }

    return '';
}

function get_staff_recent_search_history($staff_id = null)
{
    $recentSearches = get_staff_meta($staff_id ? $staff_id : get_staff_user_id(), 'recent_searches');

    if ($recentSearches == '') {
        $recentSearches = [];
    } else {
        $recentSearches = json_decode($recentSearches);
    }

    return $recentSearches;
}

function update_staff_recent_search_history($history, $staff_id = null)
{
    $totalRecentSearches = hooks()->apply_filters('total_recent_searches', 5);
    $history = array_reverse($history);
    $history = array_unique($history);
    $history = array_splice($history, 0, $totalRecentSearches);

    update_staff_meta($staff_id ? $staff_id : get_staff_user_id(), 'recent_searches', json_encode($history));

    return $history;
}


/**
 * Check if user is staff member
 * In the staff profile there is option to check IS NOT STAFF MEMBER eq like contractor
 * Some features are disabled when user is not staff member
 * @param string $staff_id staff id
 * @return boolean
 */
function is_staff_member($staff_id = '')
{
    $CI = &get_instance();
    if ($staff_id == '') {
        if (isset($GLOBALS['current_user'])) {
            return $GLOBALS['current_user']->is_not_staff === '0';
        }
        $staff_id = get_staff_user_id();
    }

    $CI->db->where('staffid', $staff_id)
        ->where('is_not_staff', 0);

    return $CI->db->count_all_results(db_prefix() . 'staff') > 0 ? true : false;
}
