<?php

	defined('BASEPATH') or exit('No direct script access allowed');

	class Proformas_model extends App_Model
	{
		const STATUS_UNPAID = 1;

		const STATUS_PAID = 2;

		const STATUS_PARTIALLY = 3;

		const STATUS_OVERDUE = 4;

		const STATUS_CANCELLED = 5;

		const STATUS_DRAFT = 6;

		private $statuses = [
			self::STATUS_UNPAID,
			self::STATUS_PAID,
			self::STATUS_PARTIALLY,
			self::STATUS_OVERDUE,
			self::STATUS_CANCELLED,
			self::STATUS_DRAFT,
		];

		private $shipping_fields = [
			'shipping_street',
			'shipping_city',
			'shipping_city',
			'shipping_state',
			'shipping_zip',
			'shipping_country',
		];

		public function __construct()
		{
			parent::__construct();
		}

		public function get_statuses()
		{
			return $this->statuses;
		}

		public function get_sale_agents()
		{
				return $this->db->query('SELECT DISTINCT(sale_agent) as sale_agent, CONCAT(firstname," ", lastname) as full_name FROM ' . db_prefix() . 'invoices JOIN ' . db_prefix() . 'staff ON ' . db_prefix() . 'staff.staffid=' . db_prefix() . 'invoices.sale_agent WHERE sale_agent != 0')->result_array();
		}

		/**
		 * Get proforma by id
		 * @param  mixed $id
		 * @return array
		 */
		public function get($id = '', $where = [])
		{
			$this->db->select('*, ' . db_prefix() . 'currencies.id as currencyid, ' . db_prefix() . 'proformas.id as id, ' . db_prefix() . 'currencies.name as currency_name');
			$this->db->select('billing_states.short_name as billing_state_name, shipping_states.short_name as shipping_state_name');
			$this->db->from(db_prefix() . 'proformas');
			$this->db->join(db_prefix() . 'currencies', '' . db_prefix() . 'currencies.id = ' . db_prefix() . 'proformas.currency', 'left');
			$this->db->join(db_prefix() . 'states billing_states', 'billing_states.state_id = ' . db_prefix() . 'proformas.billing_state', 'left');
			$this->db->join(db_prefix() . 'states shipping_states', 'shipping_states.state_id = ' . db_prefix() . 'proformas.shipping_state', 'left');

			$this->db->where($where);
			if (is_numeric($id)) {
				$this->db->where(db_prefix() . 'proformas' . '.id', $id);
				$proforma = $this->db->get()->row();
				if ($proforma) {
					$proforma->total_left_to_pay = get_proforma_total_left_to_pay($proforma->id, $proforma->total);

					$proforma->items       = get_items_by_type('proforma', $id);

					/**
					 * We keep the rates stored with the invoice
					 */
					$proforma->bnrRates = [];

					// Loop trough items and build the rates
					foreach ($proforma->items as $key => $value) {
						$proforma->bnrRates[] = daily_bnr($proforma->currency, $value['currency'], $proforma->date);
					}

					$proforma->displayRates = getRates($proforma, 'proforma');

					$proforma->attachments = $this->get_attachments($id);

					if ($proforma->project_id != 0) {
						$this->load->model('projects_model');
						$proforma->project_data = $this->projects_model->get($proforma->project_id);
					}

					$proforma->visible_attachments_to_customer_found = false;
					foreach ($proforma->attachments as $attachment) {
						if ($attachment['visible_to_customer'] == 1) {
							$proforma->visible_attachments_to_customer_found = true;

							break;
						}
					}

					$client          = $this->clients_model->get($proforma->clientid);
					$proforma->client = $client;
					if (!$proforma->client) {
						$proforma->client          = new stdClass();
						$proforma->client->company = $proforma->deleted_customer_name;
					}
					
					$this->load->model('payments_model');
					$proforma->payments = $this->payments_model->get_proforma_payments($id);

					$this->load->model('email_schedule_model');
					$proforma->scheduled_email = $this->email_schedule_model->get($id, 'proforma');


					/***proforma Currency Rates**/
					$proforma->display_currency = $this->get_display_currency($proforma);
					/***End proforma Currency Rates**/
				}

				return hooks()->apply_filters('get_proforma', $proforma);
			}

			$this->db->order_by('number,YEAR(date)', 'desc');

			return $this->db->get()->result_array();
		}


		/**
		 * @param $proforma
		 * @return array
		 */
		public function get_display_currency($proforma)
		{
			$currencies = [];
			$display_currency = '';
			foreach($proforma->items as $k=>$item_inv ){
				if($item_inv['currency_name'] == $proforma->currency_name){ continue; }
				$currencies[$item_inv['currency_name'].'_'.$item_inv['currency_rate']] = '1 '.$item_inv['currency_name'].' = '.$item_inv['currency_rate'].' '.$proforma->currency_name.'<br />';
			}

			return $currencies;
		}


		public function get_proforma_item($id)
		{
			$this->db->where('id', $id);

			return $this->db->get(db_prefix() . 'itemable')->row();
		}

		public function mark_as_cancelled($id)
		{
			$this->db->where('id', $id);
			$this->db->update(db_prefix() . 'proformas', [
				'status' => self::STATUS_CANCELLED,
				'sent'   => 1,
			]);

			if ($this->db->affected_rows() > 0) {
				$this->log_proforma_activity($id, 'proforma_activity_marked_as_cancelled');
				hooks()->do_action('proforma_marked_as_cancelled', $id);

				return true;
			}

			return false;
		}

		public function unmark_as_cancelled($id)
		{
			$this->db->where('id', $id);
			$this->db->update(db_prefix() . 'proformas', [
				'status' => self::STATUS_UNPAID,
			]);
			if ($this->db->affected_rows() > 0) {
				$this->log_proforma_activity($id, 'proforma_activity_unmarked_as_cancelled');

				return true;
			}

			return false;
		}

		/**
		 * Get this proforma generated recurring proformas
		 * @since  Version 1.0.1
		 * @param  mixed $id main proforma id
		 * @return array
		 */
		public function get_proforma_recurring_proformas($id)
		{
			$this->db->select('id');
			$this->db->where('is_recurring_from', $id);
			$proformas           = $this->db->get(db_prefix() . 'proformas')->result_array();
			$recurring_proformas = [];
			foreach ($proformas as $proforma) {
				$recurring_proformas[] = $this->get($proforma['id']);
			}

			return $recurring_proformas;
		}

		/**
		 * Get proforma total from all statuses
		 * @since  Version 1.0.2
		 * @param  mixed $data $_POST data
		 * @return array
		 */
		public function get_proformas_total($data)
		{
			$this->load->model('currencies_model');

			if (isset($data['currency'])) {
				$currencyid = $data['currency'];
			} elseif (isset($data['customer_id']) && $data['customer_id'] != '') {
				$currencyid = $this->clients_model->get_customer_default_currency($data['customer_id']);
				if ($currencyid == 0) {
					$currencyid = $this->currencies_model->get_base_currency()->id;
				}
			} elseif (isset($data['project_id']) && $data['project_id'] != '') {
				$this->load->model('projects_model');
				$currencyid = $this->projects_model->get_currency($data['project_id'])->id;
			} else {
				$currencyid = $this->currencies_model->get_base_currency()->id;
			}

			$result            = [];
			$result['due']     = [];
			$result['paid']    = [];
			$result['overdue'] = [];

			$has_permission_view                = has_permission('proformas', '', 'view');
			$has_permission_view_own            = has_permission('proformas', '', 'view_own');
			$allow_staff_view_proformas_assigned = get_option('allow_staff_view_proformas_assigned');
			$noPermissionsQuery                 = get_proformas_where_sql_for_staff(get_staff_user_id());

			for ($i = 1; $i <= 3; $i++) {
				$select = 'id,total';
				if ($i == 1) {
					$select .= ', (SELECT total - (SELECT COALESCE(SUM(amount),0) FROM ' . db_prefix() . 'proformapaymentrecords WHERE proformaid = ' . db_prefix() . 'proformas.id) - (SELECT COALESCE(SUM(amount),0) FROM ' . db_prefix() . 'credits WHERE ' . db_prefix() . 'credits.proforma_id=' . db_prefix() . 'proformas.id)) as outstanding';
				} elseif ($i == 2) {
					$select .= ',(SELECT SUM(amount) FROM ' . db_prefix() . 'proformapaymentrecords WHERE proformaid=' . db_prefix() . 'proformas.id) as total_paid';
				}
				$this->db->select($select);
				$this->db->from(db_prefix() . 'proformas');
				$this->db->where('currency', $currencyid);
				// Exclude cancelled proformas
				$this->db->where('status !=', self::STATUS_CANCELLED);
				// Exclude draft
				$this->db->where('status !=', self::STATUS_DRAFT);

				if (isset($data['project_id']) && $data['project_id'] != '') {
					$this->db->where('project_id', $data['project_id']);
				} elseif (isset($data['customer_id']) && $data['customer_id'] != '') {
					$this->db->where('clientid', $data['customer_id']);
				}

				if ($i == 3) {
					$this->db->where('status', self::STATUS_OVERDUE);
				} elseif ($i == 1) {
					$this->db->where('status !=', self::STATUS_PAID);
				}

				if (isset($data['years']) && count($data['years']) > 0) {
					$this->db->where_in('YEAR(date)', $data['years']);
				} else {
					$this->db->where('YEAR(date)', date('Y'));
				}

				if (!$has_permission_view) {
					$whereUser = $noPermissionsQuery;
					$this->db->where('(' . $whereUser . ')');
				}

				$proformas = $this->db->get()->result_array();

				foreach ($proformas as $proforma) {
					if ($i == 1) {
						$result['due'][] = $proforma['outstanding'];
					} elseif ($i == 2) {
						$result['paid'][] = $proforma['total_paid'];
					} elseif ($i == 3) {
						$result['overdue'][] = $proforma['total'];
					}
				}
			}
			$currency             = get_currency($currencyid);
			$result['due']        = array_sum($result['due']);
			$result['paid']       = array_sum($result['paid']);
			$result['overdue']    = array_sum($result['overdue']);
			$result['currency']   = $currency;
			$result['currencyid'] = $currencyid;

			return $result;
		}

		/**
		 * Insert new proforma to database
		 * @param array $data proforma data
		 * @return mixed - false if not insert, proforma ID if succes
		 */
		public function add($data, $expense = false)
		{
			unset($data['type']);
			unset($data['item_id']);
			unset($data['invoice_item_currency']);
			unset($data['invoice_item_price_converted']);
			unset($data['invoice_item_currency_rate']);
			unset($data['sku']);

			//$data['prefix'] = get_option('proforma_prefix');

			//$data['number'] = preg_replace('/[a-zA-Z]+/', '', $data['number']);
			//$data['number_format'] = get_option('proforma_number_format');
			$data['number_format'] = get_option('proforma_number_format');
			if(isset($data['s_prefix'])){
				$data['prefix'] = $data['s_prefix'];
			}
			//pre($data);

			$seria = $this->proformas_model->get_seria($data['prefix']);
			$data['number'] = (int)$seria['current_number'] + 1;

			$data['datecreated'] = date('Y-m-d H:i:s');

			$save_and_send = isset($data['save_and_send']);

			$data['addedfrom'] = !DEFINED('CRON') ? get_staff_user_id() : 0;

			$data['cancel_overdue_reminders'] = isset($data['cancel_overdue_reminders']) ? 1 : 0;

			$data['allowed_payment_modes'] = isset($data['allowed_payment_modes']) ? serialize($data['allowed_payment_modes']) : serialize([]);

			$billed_tasks = isset($data['billed_tasks']) ? array_map('unserialize', array_unique(array_map('serialize', $data['billed_tasks']))) : [];

			$billed_expenses = isset($data['billed_expenses']) ? array_map('unserialize', array_unique(array_map('serialize', $data['billed_expenses']))) : [];

			$proformas_to_merge = isset($data['proformas_to_merge']) ? $data['proformas_to_merge'] : [];

			$cancel_merged_proformas = isset($data['cancel_merged_proformas']);

			$tags = isset($data['tags']) ? $data['tags'] : '';

			if (isset($data['save_as_draft'])) {
				$data['status'] = self::STATUS_DRAFT;
				unset($data['save_as_draft']);
			} elseif (isset($data['save_and_send_later'])) {
				$data['status'] = self::STATUS_DRAFT;
				unset($data['save_and_send_later']);
			}

			if (isset($data['recurring'])) {
				if ($data['recurring'] == 'custom') {
					$data['recurring_type']   = $data['repeat_type_custom'];
					$data['custom_recurring'] = 1;
					$data['recurring']        = $data['repeat_every_custom'];
				}
			} else {
				$data['custom_recurring'] = 0;
				$data['recurring']        = 0;
			}

			if (isset($data['custom_fields'])) {
				$custom_fields = $data['custom_fields'];
				unset($data['custom_fields']);
			}

			$data['hash'] = app_generate_hash();

			$items = [];
			if (isset($data['newitems'])) {
				$items = $data['newitems'];
				unset($data['newitems']);
			}

			$data = $this->map_shipping_columns($data, $expense);

			if (isset($data['shipping_street'])) {
				$data['shipping_street'] = trim($data['shipping_street']);
				$data['shipping_street'] = nl2br($data['shipping_street']);
			}

			$data['billing_street'] = trim($data['billing_street']);
			$data['billing_street'] = nl2br($data['billing_street']);

			$hook = hooks()->apply_filters('before_proforma_added', [
				'data'  => $data,
				'items' => $items,
			]);

			$data  = $hook['data'];
			$items = $hook['items'];
			
			$this->db->insert(db_prefix() . 'proformas', $data);
			$insert_id = $this->db->insert_id();
			if ($insert_id) {
				$this->increment_next_number($data['prefix']);
				// Update next proforma number in settings
				/*$this->db->where('type', 'proforma');
				$this->db->where('name', $data['prefix']);
				$this->db->set('current_number', 'current_number+1', false);
				$this->db->update(db_prefix() . 'billing_series');*/

				if (isset($custom_fields)) {
					handle_custom_fields_post($insert_id, $custom_fields);
				}

				handle_tags_save($tags, $insert_id, 'proforma');

				foreach ($proformas_to_merge as $m) {
					$merged   = false;
					$or_merge = $this->get($m);
					if ($cancel_merged_proformas == false) {
						if ($this->delete($m, true)) {
							$merged = true;
						}
					} else {
						if ($this->mark_as_cancelled($m)) {
							$merged     = true;
							$admin_note = $or_merge->adminnote;
							$note       = 'Merged into proforma ' . format_proforma_number($insert_id);
							if ($admin_note != '') {
								$admin_note .= "\n\r" . $note;
							} else {
								$admin_note = $note;
							}
							$this->db->where('id', $m);
							$this->db->update(db_prefix() . 'proformas', [
								'adminnote' => $admin_note,
							]);
							// Delete the old items related from the merged proforma
							foreach ($or_merge->items as $or_merge_item) {
								$this->db->where('item_id', $or_merge_item['id']);
								$this->db->delete(db_prefix() . 'related_items');
							}
						}
					}
					if ($merged) {
						$this->db->where('proformaid', $or_merge->id);
						$is_expense_proforma = $this->db->get(db_prefix() . 'expenses')->row();
						if ($is_expense_proforma) {
							$this->db->where('id', $is_expense_proforma->id);
							$this->db->update(db_prefix() . 'expenses', [
								'proformaid' => $insert_id,
							]);
						}
						if (total_rows(db_prefix() . 'estimates', [
								'proformaid' => $or_merge->id,
							]) > 0) {
							$this->db->where('proformaid', $or_merge->id);
							$estimate = $this->db->get(db_prefix() . 'estimates')->row();
							$this->db->where('id', $estimate->id);
							$this->db->update(db_prefix() . 'estimates', [
								'proformaid' => $insert_id,
							]);
						} elseif (total_rows(db_prefix() . 'proposals', [
								'proforma_id' => $or_merge->id,
							]) > 0) {
							$this->db->where('proforma_id', $or_merge->id);
							$proposal = $this->db->get(db_prefix() . 'proposals')->row();
							$this->db->where('id', $proposal->id);
							$this->db->update(db_prefix() . 'proposals', [
								'proforma_id' => $insert_id,
							]);
						}
					}
				}

				foreach ($billed_tasks as $key => $tasks) {
					foreach ($tasks as $t) {
						$this->db->select('status')
										 ->where('id', $t);

						$_task = $this->db->get(db_prefix() . 'tasks')->row();

						$taskUpdateData = [
							'billed'     => 1,
							'proforma_id' => $insert_id,
						];

						if ($_task->status != Tasks_model::STATUS_COMPLETE) {
							$taskUpdateData['status']       = Tasks_model::STATUS_COMPLETE;
							$taskUpdateData['datefinished'] = date('Y-m-d H:i:s');
						}

						$this->db->where('id', $t);
						$this->db->update(db_prefix() . 'tasks', $taskUpdateData);
					}
				}

				foreach ($billed_expenses as $key => $val) {
					foreach ($val as $expense_id) {
						$this->db->where('id', $expense_id);
						$this->db->update(db_prefix() . 'expenses', [
							'proformaid' => $insert_id,
						]);
					}
				}

				update_proforma_status($insert_id);

				// Update next proforma number in settings if status is not draft
				if (!$this->is_draft($insert_id)) {
					$this->increment_next_number();
				}

				foreach ($items as $key => $item) {
					if ($itemid = add_new_sales_item_post($item, $insert_id, 'proforma')) {
						if (isset($billed_tasks[$key])) {
							foreach ($billed_tasks[$key] as $_task_id) {
								$this->db->insert(db_prefix() . 'related_items', [
									'item_id'  => $itemid,
									'rel_id'   => $_task_id,
									'rel_type' => 'task',
								]);
							}
						} elseif (isset($billed_expenses[$key])) {
							foreach ($billed_expenses[$key] as $_expense_id) {
								$this->db->insert(db_prefix() . 'related_items', [
									'item_id'  => $itemid,
									'rel_id'   => $_expense_id,
									'rel_type' => 'expense',
								]);
							}
						}
						_maybe_insert_post_item_tax($itemid, $item, $insert_id, 'proforma');
					}
				}

				update_sales_total_tax_column($insert_id, 'proforma', db_prefix() . 'proformas');

				if (!DEFINED('CRON') && $expense == false) {
					$lang_key = 'proforma_activity_created';
				} elseif (!DEFINED('CRON') && $expense == true) {
					$lang_key = 'proforma_activity_from_expense';
				} elseif (DEFINED('CRON') && $expense == false) {
					$lang_key = 'proforma_activity_recurring_created';
				} else {
					$lang_key = 'proforma_activity_recurring_from_expense_created';
				}
				$this->log_proforma_activity($insert_id, $lang_key);

				if ($save_and_send === true) {
					$this->send_proforma_to_client($insert_id, '', true, '', true);
				}
				hooks()->do_action('after_proforma_added', $insert_id);

				return $insert_id;
			}

			return false;
		}

		public function get_expenses_to_bill($clientid)
		{
			$this->load->model('expenses_model');
			$where = 'billable=1 AND clientid=' . $clientid . ' AND proformaid IS NULL';
			if (!has_permission('expenses', '', 'view')) {
				$where .= ' AND ' . db_prefix() . 'expenses.addedfrom=' . get_staff_user_id();
			}

			return $this->expenses_model->get('', $where);
		}

		public function check_for_merge_proforma($client_id, $current_proforma = '')
		{
			if ($current_proforma != '') {
				$this->db->select('status');
				$this->db->where('id', $current_proforma);
				$row = $this->db->get(db_prefix() . 'proformas')->row();
				// Cant merge on paid proforma and partialy paid and cancelled
				if ($row->status == self::STATUS_PAID || $row->status == self::STATUS_PARTIALLY || $row->status == self::STATUS_CANCELLED) {
					return [];
				}
			}

			$statuses = [
				self::STATUS_UNPAID,
				self::STATUS_OVERDUE,
				self::STATUS_DRAFT,
			];
			$noPermissionsQuery  = get_proformas_where_sql_for_staff(get_staff_user_id());
			$has_permission_view = has_permission('proformas', '', 'view');
			$this->db->select('id');
			$this->db->where('clientid', $client_id);
			$this->db->where('STATUS IN (' . implode(', ', $statuses) . ')');
			if (!$has_permission_view) {
				$whereUser = $noPermissionsQuery;
				$this->db->where('(' . $whereUser . ')');
			}
			if ($current_proforma != '') {
				$this->db->where('id !=', $current_proforma);
			}

			$proformas = $this->db->get(db_prefix() . 'proformas')->result_array();
			$proformas = hooks()->apply_filters('proformas_ids_available_for_merging', $proformas);

			$_proformas = [];

			foreach ($proformas as $proforma) {
				$inv = $this->get($proforma['id']);
				if ($inv) {
					$_proformas[] = $inv;
				}
			}

			return $_proformas;
		}

		/**
		 * Copy proforma
		 * @param  mixed $id proforma id to copy
		 * @return mixed
		 */
		public function copy($id)
		{
			$_proforma                     = $this->get($id);
			$new_proforma_data             = [];
			$new_proforma_data['clientid'] = $_proforma->clientid;
			$new_proforma_data['number']   = get_option('next_proforma_number');
			$new_proforma_data['date']     = _d(date('Y-m-d'));

			if ($_proforma->duedate && get_option('proforma_due_after') != 0) {
				$new_proforma_data['duedate'] = _d(date('Y-m-d', strtotime('+' . get_option('proforma_due_after') . ' DAY', strtotime(date('Y-m-d')))));
			}

			$new_proforma_data['save_as_draft']    = true;
			$new_proforma_data['recurring_type']   = $_proforma->recurring_type;
			$new_proforma_data['custom_recurring'] = $_proforma->custom_recurring;
			$new_proforma_data['show_quantity_as'] = $_proforma->show_quantity_as;
			$new_proforma_data['currency']         = $_proforma->currency;
			$new_proforma_data['subtotal']         = $_proforma->subtotal;
			$new_proforma_data['total']            = $_proforma->total;
			$new_proforma_data['adminnote']        = $_proforma->adminnote;
			$new_proforma_data['adjustment']       = $_proforma->adjustment;
			$new_proforma_data['discount_percent'] = $_proforma->discount_percent;
			$new_proforma_data['discount_total']   = $_proforma->discount_total;
			$new_proforma_data['recurring']        = $_proforma->recurring;
			$new_proforma_data['discount_type']    = $_proforma->discount_type;
			$new_proforma_data['terms']            = $_proforma->terms;
			$new_proforma_data['sale_agent']       = $_proforma->sale_agent;
			$new_proforma_data['project_id']       = $_proforma->project_id;
			$new_proforma_data['cycles']           = $_proforma->cycles;
			$new_proforma_data['total_cycles']     = 0;
			// Since version 1.0.6
			$new_proforma_data['billing_street']   = clear_textarea_breaks($_proforma->billing_street);
			$new_proforma_data['billing_city']     = $_proforma->billing_city;
			$new_proforma_data['billing_state']    = $_proforma->billing_state;
			$new_proforma_data['billing_zip']      = $_proforma->billing_zip;
			$new_proforma_data['billing_country']  = $_proforma->billing_country;
			$new_proforma_data['shipping_street']  = clear_textarea_breaks($_proforma->shipping_street);
			$new_proforma_data['shipping_city']    = $_proforma->shipping_city;
			$new_proforma_data['shipping_state']   = $_proforma->shipping_state;
			$new_proforma_data['shipping_zip']     = $_proforma->shipping_zip;
			$new_proforma_data['shipping_country'] = $_proforma->shipping_country;
			if ($_proforma->include_shipping == 1) {
				$new_proforma_data['include_shipping'] = $_proforma->include_shipping;
			}
			$new_proforma_data['show_shipping_on_invoice'] = $_proforma->show_shipping_on_invoice;
			// Set to unpaid status automatically
			$new_proforma_data['status']                = self::STATUS_UNPAID;
			$new_proforma_data['clientnote']            = $_proforma->clientnote;
			$new_proforma_data['adminnote']             = $_proforma->adminnote;
			$new_proforma_data['allowed_payment_modes'] = unserialize($_proforma->allowed_payment_modes);
			$new_proforma_data['newitems']              = [];
			$key                                       = 1;

			$custom_fields_items = get_custom_fields('items');
			foreach ($_proforma->items as $item) {
				$new_proforma_data['newitems'][$key]['description']      = $item['description'];
				$new_proforma_data['newitems'][$key]['long_description'] = clear_textarea_breaks($item['long_description']);
				$new_proforma_data['newitems'][$key]['qty']              = $item['qty'];
				$new_proforma_data['newitems'][$key]['unit']             = $item['unit'];
				$new_proforma_data['newitems'][$key]['taxname']          = [];
				$taxes                                                  = get_proforma_item_taxes($item['id']);
				foreach ($taxes as $tax) {
					// tax name is in format TAX1|10.00
					array_push($new_proforma_data['newitems'][$key]['taxname'], $tax['taxname']);
				}
				$new_proforma_data['newitems'][$key]['rate']  = $item['rate'];
				$new_proforma_data['newitems'][$key]['order'] = $item['item_order'];

				foreach ($custom_fields_items as $cf) {
					$new_proforma_data['newitems'][$key]['custom_fields']['items'][$cf['id']] = get_custom_field_value($item['id'], $cf['id'], 'items', false);

					if (!defined('COPY_CUSTOM_FIELDS_LIKE_HANDLE_POST')) {
						define('COPY_CUSTOM_FIELDS_LIKE_HANDLE_POST', true);
					}
				}
				$key++;
			}
			$id = $this->proformas_model->add($new_proforma_data);
			if ($id) {
				$this->db->where('id', $id);
				$this->db->update(db_prefix() . 'proformas', [
					'cancel_overdue_reminders' => $_proforma->cancel_overdue_reminders,
				]);

				$custom_fields = get_custom_fields('proforma');
				foreach ($custom_fields as $field) {
					$value = get_custom_field_value($_proforma->id, $field['id'], 'proforma', false);
					if ($value == '') {
						continue;
					}
					$this->db->insert(db_prefix() . 'customfieldsvalues', [
						'relid'   => $id,
						'fieldid' => $field['id'],
						'fieldto' => 'proforma',
						'value'   => $value,
					]);
				}

				$tags = get_tags_in($_proforma->id, 'proforma');
				handle_tags_save($tags, $id, 'proforma');

				log_activity('Copied Invoice ' . format_proforma_number($_proforma->id));

				hooks()->do_action('proforma_copied', ['copy_from' => $_proforma->id, 'copy_id' => $id]);

				return $id;
			}

			return false;
		}

		/**
		 * Update proforma data
		 * @param  array $data proforma data
		 * @param  mixed $id   proformaid
		 * @return boolean
		 */
		public function update($data, $id)
		{

			unset($data['type']);
			unset($data['item_id']);
			unset($data['invoice_item_currency']);
			unset($data['invoice_item_price_converted']);
			unset($data['invoice_item_currency_rate']);
			unset($data['sku']);

			$original_proforma = $this->get($id);
			$affectedRows     = 0;
			if (isset($data['nubmer'])) {
					$data['number'] = trim($data['number']);
			}

			$original_number_formatted = format_proforma_number($id);

			$original_number = $original_proforma->number;

			$save_and_send = isset($data['save_and_send']);

			$cancel_merged_proformas = isset($data['cancel_merged_proformas']);

			$proformas_to_merge = isset($data['proformas_to_merge']) ? $data['proformas_to_merge'] : [];

			$billed_tasks = isset($data['billed_tasks']) ? $data['billed_tasks'] : [];

			$billed_expenses = isset($data['billed_expenses']) ? array_map('unserialize', array_unique(array_map('serialize', $data['billed_expenses']))) : [];

			$data['cancel_overdue_reminders'] = isset($data['cancel_overdue_reminders']) ? 1 : 0;

			$data['cycles'] = !isset($data['cycles']) ? 0 : $data['cycles'];

			$data['allowed_payment_modes'] = isset($data['allowed_payment_modes']) ? serialize($data['allowed_payment_modes']) : serialize([]);

			// Recurring proforma set to NO, Cancelled
			if ($original_proforma->recurring != 0 && $data['repeat_every_custom'] == 0) {
				$data['cycles']              = 0;
				$data['total_cycles']        = 0;
				$data['last_recurring_date'] = null;
			}
			
			if (isset($data['recurring'])) {
				if ($data['recurring'] == 'custom') {
					$data['recurring_type']   = $data['repeat_type_custom'];
					$data['custom_recurring'] = 1;
					$data['recurring']        = $data['repeat_every_custom'];
				} else {
					$data['recurring_type']   = null;
					$data['custom_recurring'] = 0;
				}
			} else {
				$data['custom_recurring'] = 0;
				$data['recurring']        = 0;
				$data['recurring_type']   = null;
			}

			$items = [];
			if (isset($data['items'])) {
				$items = $data['items'];
				unset($data['items']);
			}

			$newitems = [];
			if (isset($data['newitems'])) {
				$newitems = $data['newitems'];
				unset($data['newitems']);
			}

			if (isset($data['custom_fields'])) {
				$custom_fields = $data['custom_fields'];
				if (handle_custom_fields_post($id, $custom_fields)) {
					$affectedRows++;
				}
				unset($data['custom_fields']);
			}

			if (isset($data['tags'])) {
				if (handle_tags_save($data['tags'], $id, 'proforma')) {
					$affectedRows++;
				}
			}

			$data = $this->map_shipping_columns($data);

			$data['billing_street']  = trim($data['billing_street']);
			$data['shipping_street'] = trim($data['shipping_street']);

			$data['billing_street']  = nl2br($data['billing_street']);
			$data['shipping_street'] = nl2br($data['shipping_street']);

			$hook = hooks()->apply_filters('before_proforma_updated', [
				'data'          => $data,
				'items'         => $items,
				'newitems'      => $newitems,
				'removed_items' => isset($data['removed_items']) ? $data['removed_items'] : [],
			], $id);

			$data                  = $hook['data'];
			$items                 = $hook['items'];
			$newitems              = $hook['newitems'];
			$data['removed_items'] = $hook['removed_items'];

			foreach ($billed_tasks as $key => $tasks) {
				foreach ($tasks as $t) {
					$this->db->select('status')
									 ->where('id', $t);
					$_task          = $this->db->get(db_prefix() . 'tasks')->row();
					$taskUpdateData = [
						'billed'     => 1,
						'proforma_id' => $id,
					];
					if ($_task->status != Tasks_model::STATUS_COMPLETE) {
						$taskUpdateData['status']       = Tasks_model::STATUS_COMPLETE;
						$taskUpdateData['datefinished'] = date('Y-m-d H:i:s');
					}
					$this->db->where('id', $t);
					$this->db->update(db_prefix() . 'tasks', $taskUpdateData);
				}
			}

			foreach ($billed_expenses as $key => $val) {
				foreach ($val as $expense_id) {
					$this->db->where('id', $expense_id);
					$this->db->update(db_prefix() . 'expenses', [
						'proformaid' => $id,
					]);
				}
			}

			// Delete items checked to be removed from database
			foreach ($data['removed_items'] as $remove_item_id) {
				$original_item = $this->get_proforma_item($remove_item_id);
				if (handle_removed_sales_item_post($remove_item_id, 'proforma')) {
					$affectedRows++;

					$this->log_proforma_activity($id, 'proforma_estimate_activity_removed_item', false, serialize([
						$original_item->description,
					]));

					$this->db->where('item_id', $original_item->id);
					$related_items = $this->db->get(db_prefix() . 'related_items')->result_array();

					foreach ($related_items as $rel_item) {
						if ($rel_item['rel_type'] == 'task') {
							$this->db->where('id', $rel_item['rel_id']);
							$this->db->update(db_prefix() . 'tasks', [
								'proforma_id' => null,
								'billed'     => 0,
							]);
						} elseif ($rel_item['rel_type'] == 'expense') {
							$this->db->where('id', $rel_item['rel_id']);
							$this->db->update(db_prefix() . 'expenses', [
								'proformaid' => null,
							]);
						}
						$this->db->where('item_id', $original_item->id);
						$this->db->delete(db_prefix() . 'related_items');
					}
				}
			}

			unset($data['removed_items']);

			$this->db->where('id', $id);
			$this->db->update(db_prefix() . 'proformas', $data);
			//var_dump($data);
			if ($this->db->affected_rows() > 0) {
				$affectedRows++;
				if (isset($data['number']) && $original_number != $data['number']) {
					$this->log_proforma_activity($original_proforma->id, 'proforma_activity_number_changed', false, serialize([
						$original_number_formatted,
						format_proforma_number($original_proforma->id),
					]));
				}
			}

			if (count($items) > 0) {
				foreach ($items as $key => $item) {
					$original_item = $this->get_proforma_item($item['itemid']);

					if (update_sales_item_post($item['itemid'], $item, 'item_order')) {
						$affectedRows++;
					}

					if (update_sales_item_post($item['itemid'], $item, 'unit')) {
						$affectedRows++;
					}

					if (update_sales_item_post($item['itemid'], $item, 'description')) {
						$this->log_proforma_activity($id, 'proforma_estimate_activity_updated_item_short_description', false, serialize([
							$original_item->description,
							$item['description'],
						]));
						$affectedRows++;
					}

					if (update_sales_item_post($item['itemid'], $item, 'long_description')) {
						$this->log_proforma_activity($id, 'proforma_estimate_activity_updated_item_long_description', false, serialize([
							$original_item->long_description,
							$item['long_description'],
						]));
						$affectedRows++;
					}

					if (update_sales_item_post($item['itemid'], $item, 'rate')) {
						$this->log_proforma_activity($id, 'proforma_estimate_activity_updated_item_rate', false, serialize([
							$original_item->rate,
							$item['rate'],
						]));
						$affectedRows++;
					}

					if (update_sales_item_post($item['itemid'], $item, 'qty')) {
						$this->log_proforma_activity($id, 'proforma_estimate_activity_updated_qty_item', false, serialize([
							$item['description'],
							$original_item->qty,
							$item['qty'],
						]));
						$affectedRows++;
					}

					if (update_sales_item_post($item['itemid'], $item, 'currency')) {
						/*$this->log_invoice_activity($id, 'invoice_estimate_activity_updated_qty_item', false, serialize([
							$item['description'],
							$original_item->qty,
							$item['qty'],
						]));*/
						$affectedRows++;
					}
					if (update_sales_item_post($item['itemid'], $item, 'sku')) {
						$affectedRows++;
					}
					if (update_sales_item_post($item['itemid'], $item, 'price_converted')) {
						$affectedRows++;
					}
					if (update_sales_item_post($item['itemid'], $item, 'currency_rate')) {
						$affectedRows++;
					}

					if (isset($item['custom_fields'])) {
						if (handle_custom_fields_post($item['itemid'], $item['custom_fields'])) {
							$affectedRows++;
						}
					}

					if (!isset($item['taxname']) || (isset($item['taxname']) && count($item['taxname']) == 0)) {
						if (delete_taxes_from_item($item['itemid'], 'proforma')) {
							$affectedRows++;
						}
					} else {
						$item_taxes        = get_proforma_item_taxes($item['itemid']);
						$_item_taxes_names = [];
						foreach ($item_taxes as $_item_tax) {
							array_push($_item_taxes_names, $_item_tax['taxname']);
						}
						$i = 0;
						foreach ($_item_taxes_names as $_item_tax) {
							if (!in_array($_item_tax, $item['taxname'])) {
								$this->db->where('id', $item_taxes[$i]['id'])
												 ->delete(db_prefix() . 'item_tax');
								if ($this->db->affected_rows() > 0) {
									$affectedRows++;
								}
							}
							$i++;
						}
						if (_maybe_insert_post_item_tax($item['itemid'], $item, $id, 'proforma')) {
							$affectedRows++;
						}
					}
				}
			}

			foreach ($newitems as $key => $item) {
				if ($new_item_added = add_new_sales_item_post($item, $id, 'proforma')) {
					if (isset($billed_tasks[$key])) {
						foreach ($billed_tasks[$key] as $_task_id) {
							$this->db->insert(db_prefix() . 'related_items', [
								'item_id'  => $new_item_added,
								'rel_id'   => $_task_id,
								'rel_type' => 'task',
							]);
						}
					} elseif (isset($billed_expenses[$key])) {
						foreach ($billed_expenses[$key] as $_expense_id) {
							$this->db->insert(db_prefix() . 'related_items', [
								'item_id'  => $new_item_added,
								'rel_id'   => $_expense_id,
								'rel_type' => 'expense',
							]);
						}
					}
					_maybe_insert_post_item_tax($new_item_added, $item, $id, 'proforma');

					$this->log_proforma_activity($id, 'proforma_estimate_activity_added_item', false, serialize([
						$item['description'],
					]));
					$affectedRows++;
				}
			}

			foreach ($proformas_to_merge as $m) {
				$merged   = false;
				$or_merge = $this->get($m);
				if ($cancel_merged_proformas == false) {
					if ($this->delete($m, true)) {
						$merged = true;
					}
				} else {
					if ($this->mark_as_cancelled($m)) {
						$merged     = true;
						$admin_note = $or_merge->adminnote;
						$note       = 'Merged into proforma ' . format_proforma_number($id);
						if ($admin_note != '') {
							$admin_note .= "\n\r" . $note;
						} else {
							$admin_note = $note;
						}
						$this->db->where('id', $m);
						$this->db->update(db_prefix() . 'proformas', [
							'adminnote' => $admin_note,
						]);
					}
				}
				if ($merged) {
					$this->db->where('proformaid', $or_merge->id);
					$is_expense_proforma = $this->db->get(db_prefix() . 'expenses')->row();
					if ($is_expense_proforma) {
						$this->db->where('id', $is_expense_proforma->id);
						$this->db->update(db_prefix() . 'expenses', [
							'proformaid' => $id,
						]);
					}
					if (total_rows(db_prefix() . 'estimates', [
							'proformaid' => $or_merge->id,
						]) > 0) {
						$this->db->where('proformaid', $or_merge->id);
						$estimate = $this->db->get(db_prefix() . 'estimates')->row();
						$this->db->where('id', $estimate->id);
						$this->db->update(db_prefix() . 'estimates', [
							'proformaid' => $id,
						]);
					} elseif (total_rows(db_prefix() . 'proposals', [
							'proforma_id' => $or_merge->id,
						]) > 0) {
						$this->db->where('proforma_id', $or_merge->id);
						$proposal = $this->db->get(db_prefix() . 'proposals')->row();
						$this->db->where('id', $proposal->id);
						$this->db->update(db_prefix() . 'proposals', [
							'proforma_id' => $id,
						]);
					}
				}
			}
			update_sales_total_tax_column($id, 'proforma', db_prefix() . 'proformas');
			if ($affectedRows > 0) {
				update_proforma_status($id);
			}

			if ($save_and_send === true) {
				$this->send_proforma_to_client($id, '', true, '', true);
			}

			if ($affectedRows > 0) {
				hooks()->do_action('after_proforma_updated', $id);

				return true;
			}

			return false;
		}

		public function get_attachments($proformaid, $id = '')
		{
			// If is passed id get return only 1 attachment
			if (is_numeric($id)) {
				$this->db->where('id', $id);
			} else {
				$this->db->where('rel_id', $proformaid);
			}

			$this->db->where('rel_type', 'proforma');
			$result = $this->db->get(db_prefix() . 'files');
			if (is_numeric($id)) {
				return $result->row();
			}

			return $result->result_array();
		}

		/**
		 *  Delete proforma attachment
		 * @since  Version 1.0.4
		 * @param   mixed $id  attachmentid
		 * @return  boolean
		 */
		public function delete_attachment($id)
		{
			$attachment = $this->get_attachments('', $id);
			$deleted    = false;
			if ($attachment) {
				if (empty($attachment->external)) {
					unlink(get_upload_path_by_type('proforma') . $attachment->rel_id . '/' . $attachment->file_name);
				}
				$this->db->where('id', $attachment->id);
				$this->db->delete(db_prefix() . 'files');
				if ($this->db->affected_rows() > 0) {
					$deleted = true;
					log_activity('Invoice Attachment Deleted [InvoiceID: ' . $attachment->rel_id . ']');
				}

				if (is_dir(get_upload_path_by_type('proforma') . $attachment->rel_id)) {
					// Check if no attachments left, so we can delete the folder also
					$other_attachments = list_files(get_upload_path_by_type('proforma') . $attachment->rel_id);
					if (count($other_attachments) == 0) {
						// okey only index.html so we can delete the folder also
						delete_dir(get_upload_path_by_type('proforma') . $attachment->rel_id);
					}
				}
			}

			return $deleted;
		}

		/**
		 * Delete proforma items and all connections
		 * @param  mixed $id proformaid
		 * @return boolean
		 */
		public function delete($id, $simpleDelete = false)
		{
			if (get_option('delete_only_on_last_proforma') == 1 && $simpleDelete == false) {
				if (!is_last_proforma($id)) {
					return false;
				}
			}
			$number = format_proforma_number($id);

			hooks()->do_action('before_proforma_deleted', $id);
			$this->db->where('id', $id);
			$this->db->delete(db_prefix() . 'proformas');
			if ($this->db->affected_rows() > 0) {
				if (get_option('proforma_number_decrement_on_delete') == 1 && $simpleDelete == false) {
					$current_next_proforma_number = get_option('next_proforma_number');
					if ($current_next_proforma_number > 1) {
						// Decrement next proforma number to
						$this->db->where('name', 'next_proforma_number');
						$this->db->set('value', 'value-1', false);
						$this->db->update(db_prefix() . 'options');
					}
				}
				if ($simpleDelete == false) {
					$this->db->where('proformaid', $id);
					$this->db->update(db_prefix() . 'expenses', [
						'proformaid' => null,
					]);

					$this->db->where('proforma_id', $id);
					$this->db->update(db_prefix() . 'proposals', [
						'proforma_id'     => null,
						'date_converted' => null,
					]);

					$this->db->where('proforma_id', $id);
					$this->db->update(db_prefix() . 'tasks', [
						'proforma_id' => null,
						'billed'     => 0,
					]);

					// if is converted from estimate set the estimate proforma to null
					if (total_rows(db_prefix() . 'estimates', [
							'proformaid' => $id,
						]) > 0) {
						$this->db->where('proformaid', $id);
						$estimate = $this->db->get(db_prefix() . 'estimates')->row();
						$this->db->where('id', $estimate->id);
						$this->db->update(db_prefix() . 'estimates', [
							'proformaid'     => null,
							'proformad_date' => null,
						]);
						$this->load->model('estimates_model');
						$this->estimates_model->log_estimate_activity($estimate->id, 'not_estimate_proforma_deleted');
					}
				}

				$this->db->select('id');
				$this->db->where('id IN (SELECT credit_id FROM ' . db_prefix() . 'credits WHERE proforma_id=' . $this->db->escape_str($id) . ')');
				$linked_credit_notes = $this->db->get(db_prefix() . 'creditnotes')->result_array();

				$this->db->where('proforma_id', $id);
				$this->db->delete(db_prefix() . 'credits');

				$this->load->model('credit_notes_model');
				foreach ($linked_credit_notes as $credit_note) {
					$this->credit_notes_model->update_credit_note_status($credit_note['id']);
				}

				$this->db->where('rel_id', $id);
				$this->db->where('rel_type', 'proforma');
				$this->db->delete(db_prefix() . 'notes');

				delete_tracked_emails($id, 'proforma');

				$this->db->where('rel_type', 'proforma');
				$this->db->where('rel_id', $id);
				$this->db->delete(db_prefix() . 'taggables');

				$this->db->where('rel_type', 'proforma');
				$this->db->where('rel_id', $id);
				$this->db->delete(db_prefix() . 'reminders');

				$this->db->where('rel_type', 'proforma');
				$this->db->where('rel_id', $id);
				$this->db->delete(db_prefix() . 'views_tracking');

				$this->db->where('relid IN (SELECT id from ' . db_prefix() . 'itemable WHERE rel_type="proforma" AND rel_id="' . $this->db->escape_str($id) . '")');
				$this->db->where('fieldto', 'items');
				$this->db->delete(db_prefix() . 'customfieldsvalues');

				$items = get_items_by_type('proforma', $id);

				$this->db->where('rel_id', $id);
				$this->db->where('rel_type', 'proforma');
				$this->db->delete(db_prefix() . 'itemable');

				foreach ($items as $item) {
					$this->db->where('item_id', $item['id']);
					$this->db->delete(db_prefix() . 'related_items');
				}

				$this->db->where('proformaid', $id);
				$this->db->delete(db_prefix() . 'proformapaymentrecords');

				$this->db->where('rel_id', $id);
				$this->db->where('rel_type', 'proforma');
				$this->db->delete(db_prefix() . 'sales_activity');

				$this->db->where('is_recurring_from', $id);
				$this->db->update(db_prefix() . 'proformas', [
					'is_recurring_from' => null,
				]);

				// Delete the custom field values
				$this->db->where('relid', $id);
				$this->db->where('fieldto', 'proforma');
				$this->db->delete(db_prefix() . 'customfieldsvalues');

				$this->db->where('rel_id', $id);
				$this->db->where('rel_type', 'proforma');
				$this->db->delete(db_prefix() . 'item_tax');

				// Get billed tasks for this proforma and set to unbilled
				$this->db->where('proforma_id', $id);
				$tasks = $this->db->get(db_prefix() . 'tasks')->result_array();
				foreach ($tasks as $task) {
					$this->db->where('id', $task['id']);
					$this->db->update(db_prefix() . 'tasks', [
						'proforma_id' => null,
						'billed'     => 0,
					]);
				}

				$attachments = $this->get_attachments($id);
				foreach ($attachments as $attachment) {
					$this->delete_attachment($attachment['id']);
				}
				// Get related tasks
				$this->db->where('rel_type', 'proforma');
				$this->db->where('rel_id', $id);
				$tasks = $this->db->get(db_prefix() . 'tasks')->result_array();
				foreach ($tasks as $task) {
					$this->tasks_model->delete_task($task['id']);
				}
				if ($simpleDelete == false) {
					log_activity('Invoice Deleted [' . $number . ']');
				}

				return true;
			}

			return false;
		}

		/**
		 * Set proforma to sent when email is successfuly sended to client
		 * @param mixed $id proformaid
		 * @param  mixed $manually is staff manually marking this proforma as sent
		 * @return  boolean
		 */
		public function set_proforma_sent($id, $manually = false, $emails_sent = [], $is_status_updated = false)
		{
			$this->db->where('id', $id);
			$this->db->update(db_prefix() . 'proformas', [
				'sent'     => 1,
				'datesend' => date('Y-m-d H:i:s'),
			]);
			$marked = false;
			if ($this->db->affected_rows() > 0) {
				$marked = true;
			}
			if (DEFINED('CRON')) {
				$additional_activity_data = serialize([
					'<custom_data>' . implode(', ', $emails_sent) . '</custom_data>',
				]);
				$description = 'proforma_activity_sent_to_client_cron';
			} else {
				if ($manually == false) {
					$additional_activity_data = serialize([
						'<custom_data>' . implode(', ', $emails_sent) . '</custom_data>',
					]);
					$description = 'proforma_activity_sent_to_client';
				} else {
					$additional_activity_data = serialize([]);
					$description              = 'proforma_activity_marked_as_sent';
				}
			}

			if ($is_status_updated == false) {
				update_proforma_status($id, true);
			}

			$this->db->where('rel_id', $id);
			$this->db->where('rel_type', 'proforma');
			$this->db->delete('scheduled_emails');

			$this->log_proforma_activity($id, $description, false, $additional_activity_data);

			return $marked;
		}

		/**
		 * Send overdue notice to client for this proforma
		 * @param  mxied  $id   proformaid
		 * @return boolean
		 */
		public function send_proforma_overdue_notice($id)
		{
			$proforma        = $this->get($id);
			$proforma_number = format_proforma_number($proforma->id);
			set_mailing_constant();
			$pdf = proforma_pdf($proforma);

			$attach_pdf = hooks()->apply_filters('proforma_overdue_notice_attach_pdf', true);

			if ($attach_pdf === true) {
				$attach = $pdf->Output($proforma_number . '.pdf', 'S');
			}

			$emails_sent      = [];
			$email_sent       = false;
			$sms_sent         = false;
			$sms_reminder_log = [];

			// For all cases update this to prevent sending multiple reminders eq on fail
			$this->db->where('id', $id);
			$this->db->update(db_prefix() . 'proformas', [
				'last_overdue_reminder' => date('Y-m-d'),
			]);

			$contacts = $this->clients_model->get_contacts($proforma->clientid, ['active' => 1, 'proforma_emails' => 1]);
			foreach ($contacts as $contact) {
				$template = mail_template('proforma_overdue_notice', $proforma, $contact);

				if ($attach_pdf === true) {
					$template->add_attachment([
						'attachment' => $attach,
						'filename'   => str_replace('/', '-', $proforma_number . '.pdf'),
						'type'       => 'application/pdf',
					]);
				}

				$merge_fields = $template->get_merge_fields();

				if ($template->send()) {
					array_push($emails_sent, $contact['email']);
					$email_sent = true;
				}

				if (can_send_sms_based_on_creation_date($proforma->datecreated)
					&& $this->app_sms->trigger(SMS_TRIGGER_INVOICE_OVERDUE, $contact['phonenumber'], $merge_fields)) {
					$sms_sent = true;
					array_push($sms_reminder_log, $contact['firstname'] . ' (' . $contact['phonenumber'] . ')');
				}
			}

			if ($email_sent || $sms_sent) {
				if ($email_sent) {
					$this->log_proforma_activity($id, 'user_sent_overdue_reminder', false, serialize([
						'<custom_data>' . implode(', ', $emails_sent) . '</custom_data>',
						defined('CRON') ? ' ' : get_staff_full_name(),
					]));
				}

				if ($sms_sent) {
					$this->log_proforma_activity($id, 'sms_reminder_sent_to', false, serialize([
						implode(', ', $sms_reminder_log),
					]));
				}

				hooks()->do_action('proforma_overdue_reminder_sent', [
					'proforma_id' => $id,
					'sent_to'    => $emails_sent,
					'sms_send'   => $sms_sent,
				]);

				return true;
			}

			return false;
		}

		/**
		 * Send proforma to client
		 * @param  mixed  $id        proformaid
		 * @param  string  $template  email template to sent
		 * @param  boolean $attachpdf attach proforma pdf or not
		 * @return boolean
		 */
		public function send_proforma_to_client($id, $template_name = '', $attachpdf = true, $cc = '', $manually = false, $attachStatement = [])
		{
			$proforma = $this->get($id);

			$proforma = hooks()->apply_filters('proforma_object_before_send_to_client', $proforma);

			if ($template_name == '') {
				if ($proforma->sent == 0) {
					$template_name = 'proforma_send_to_customer';
				} else {
					$template_name = 'proforma_send_to_customer_already_sent';
				}

				$template_name = hooks()->apply_filters('after_proforma_sent_template_statement', $template_name);
			}

			$proforma_number = format_proforma_number($proforma->id);
			$emails_sent    = [];
			$sent           = false;
			$sent_to        = [];

			// Manually is used when sending the proforma via add/edit area button Save & Send
			if (!DEFINED('CRON') && $manually === false) {
				$sent_to = $this->input->post('sent_to');
			} elseif (isset($GLOBALS['scheduled_email_contacts'])) {
				$sent_to = $GLOBALS['scheduled_email_contacts'];
			} else {
				$contacts = $this->clients_model->get_contacts($proforma->clientid, ['active' => 1, 'proforma_emails' => 1]);

				foreach ($contacts as $contact) {
					array_push($sent_to, $contact['id']);
				}
			}

			$attachStatementPdf = false;
			if (is_array($sent_to) && count($sent_to) > 0) {
				if (isset($attachStatement['attach']) && $attachStatement['attach'] == true) {
					$statement    = $this->clients_model->get_statement($proforma->clientid, $attachStatement['from'], $attachStatement['to']);
					$statementPdf = statement_pdf($statement);

					$statementPdfFileName = slug_it(_l('customer_statement') . '-' . $statement['client']->company);

					$attachStatementPdf = $statementPdf->Output($statementPdfFileName . '.pdf', 'S');
				}

				$status_updated = update_proforma_status($proforma->id, true, true);

				if ($attachpdf) {
					$_pdf_proforma = $this->get($id);
					set_mailing_constant();
					$pdf    = proforma_pdf($_pdf_proforma);
					$attach = $pdf->Output($proforma_number . '.pdf', 'S');
				}

				$i = 0;
				foreach ($sent_to as $contact_id) {
					if ($contact_id != '') {

						// Send cc only for the first contact
						if (!empty($cc) && $i > 0) {
							$cc = '';
						}

						$contact = $this->clients_model->get_contact($contact_id);

						if (!$contact) {
							continue;
						}

						$template = mail_template($template_name, $proforma, $contact, $cc);

						if ($attachpdf) {
							$template->add_attachment([
								'attachment' => $attach,
								'filename'   => str_replace('/', '-', $proforma_number . '.pdf'),
								'type'       => 'application/pdf',
							]);
						}

						if ($attachStatementPdf) {
							$template->add_attachment([
								'attachment' => $attachStatementPdf,
								'filename'   => $statementPdfFileName,
								'type'       => 'application/pdf',
							]);
						}

						if ($template->send()) {
							$sent = true;
							array_push($emails_sent, $contact->email);
						}
					}
					$i++;
				}
			} else {
				return false;
			}

			if ($sent) {
				$this->set_proforma_sent($id, false, $emails_sent, true);
				hooks()->do_action('proforma_sent', $id);

				return true;
			}

			// In case the proforma not sended and the status was draft and the proforma status is updated before send return back to draft status
			if ($proforma->status == self::STATUS_DRAFT && $status_updated !== false) {
				$this->db->where('id', $proforma->id);
				$this->db->update(db_prefix() . 'proformas', [
					'status' => self::STATUS_DRAFT,
				]);
			}

			return false;
		}

		/**
		 * All proforma activity
		 * @param  mixed $id proformaid
		 * @return array
		 */
		public function get_proforma_activity($id)
		{
			$this->db->where('rel_id', $id);
			$this->db->where('rel_type', 'proforma');
			$this->db->order_by('date', 'asc');

			return $this->db->get(db_prefix() . 'sales_activity')->result_array();
		}

		/**
		 * Log proforma activity to database
		 * @param  mixed $id   proformaid
		 * @param  string $description activity description
		 */
		public function log_proforma_activity($id, $description = '', $client = false, $additional_data = '')
		{
			if (DEFINED('CRON')) {
				$staffid   = '[CRON]';
				$full_name = '[CRON]';
			} elseif (defined('STRIPE_SUBSCRIPTION_INVOICE')) {
				$staffid   = null;
				$full_name = '[Stripe]';
			} elseif ($client == true) {
				$staffid   = null;
				$full_name = '';
			} else {
				$staffid   = get_staff_user_id();
				$full_name = get_staff_full_name(get_staff_user_id());
			}
			$this->db->insert(db_prefix() . 'sales_activity', [
				'description'     => $description,
				'date'            => date('Y-m-d H:i:s'),
				'rel_id'          => $id,
				'rel_type'        => 'proforma',
				'staffid'         => $staffid,
				'full_name'       => $full_name,
				'additional_data' => $additional_data,
			]);
		}

		public function get_proformas_years()
		{
			return $this->db->query('SELECT DISTINCT(YEAR(date)) as year FROM ' . db_prefix() . 'proformas ORDER BY year DESC')->result_array();
		}

		private function map_shipping_columns($data, $expense = false)
		{
			if (!isset($data['include_shipping'])) {
				foreach ($this->shipping_fields as $_s_field) {
					if (isset($data[$_s_field])) {
						$data[$_s_field] = null;
					}
				}
				$data['show_shipping_on_invoice'] = 1;
				$data['include_shipping']         = 0;
			} else {
				// We dont need to overwrite to 1 unless its coming from the main function add
				if (!DEFINED('CRON') && $expense == false) {
					$data['include_shipping'] = 1;
					// set by default for the next time to be checked
					if (isset($data['show_shipping_on_invoice']) && ($data['show_shipping_on_invoice'] == 1 || $data['show_shipping_on_invoice'] == 'on')) {
						$data['show_shipping_on_invoice'] = 1;
					} else {
						$data['show_shipping_on_invoice'] = 0;
					}
				}
				// else its just like they are passed
			}

			return $data;
		}

		public function getNextNumber($serie) {
			$this->db->where('type', 'proforma');
			$this->db->where('name', $serie);
			$row = $this->db->get(db_prefix() . 'billing_series')->row();

			return $row->current_number;
		}

    /**
     * @since  2.7.0
     *
     * Change the proforma number for status when it's draft
     *
     * @param int $id
     *
     * @return int
     */
    public function change_proforma_number_when_status_draft($id)
    {
        $this->db->select('number')->where('id', $id);
        $proforma = $this->db->get('proformas')->row();

        $newNumber = get_option('next_proforma_number');

        $this->db->where('id', $id);
        $this->db->update('proformas', ['number' => $newNumber]);

        $this->increment_next_number();

        return $proforma->number;
    }

    /**
     * @since  2.7.0
     *
     * Increment the invoies next nubmer
     *
     * @return void
     */
    public function increment_next_number_old()
    {
        // Update next proforma number in settings
        $this->db->where('name', 'next_proforma_number');
        $this->db->set('value', 'value+1', false);
        $this->db->update(db_prefix() . 'options');
    }

    /**
     * @since  2.7.0
     *
     * Check whether the given invoice is draft
     *
     * @param int $id
     *
     * @return boolean
     */
    public function is_draft($id)
    {
        return total_rows('proformas', ['id' => $id, 'status' => self::STATUS_DRAFT]) === 1;
    }

    /**
     * @since  2.7.0
     *
     * Decrement the invoies next number
     *
     * @return void
     */
    public function decrement_next_number_old()
    {
        $this->db->where('name', 'next_proforma_number');
        $this->db->set('value', 'value-1', false);
        $this->db->update(db_prefix() . 'options');
    }

		public function convert_invoice($id) {
			$proforma = $this->get($id);

			//$this->load->model('invoices_model');
			//$serie = $this->invoices_model->getDefaultSerie();

			$invoice             = [];
			$invoice['clientid'] = $proforma->clientid;
		//	$invoice['number']   = $serie->name.$serie->current_number;

		$this->load->model('invoices_model');

			$seria = $this->invoices_model->get_seria(); //$data['prefix']
			if(!$seria){
				set_alert('warning', _l('no_default_billling_series_in_account', _l('billing')));
				redirect(admin_url('billing/config/series' ));
			}
			//$invoice['number'] = (int)$seria['current_number'] + 1;
			$invoice['prefix'] = $seria['name'];
			$invoice['date']     = _d(date('Y-m-d'));

			if ($proforma->duedate && get_option('invoice_due_after') != 0) {
				$invoice['duedate'] = _d(date('Y-m-d', strtotime('+' . get_option('invoice_due_after') . ' DAY', strtotime(date('Y-m-d')))));
			}

			//$invoice['save_as_draft']    = false;
			//$invoice['save_and_send_later'] = true;
			$invoice['recurring_type']   = $proforma->recurring_type;
			$invoice['custom_recurring'] = $proforma->custom_recurring;
			$invoice['show_quantity_as'] = $proforma->show_quantity_as;
			$invoice['currency']         = $proforma->currency;
			$invoice['subtotal']         = $proforma->subtotal;
			$invoice['total']            = $proforma->total;
			$invoice['adminnote']        = $proforma->adminnote;
			$invoice['adjustment']       = $proforma->adjustment;
			$invoice['discount_percent'] = $proforma->discount_percent;
			$invoice['discount_total']   = $proforma->discount_total;
			$invoice['recurring']        = $proforma->recurring;
			$invoice['discount_type']    = $proforma->discount_type;
			$invoice['terms']            = $proforma->terms;
			$invoice['sale_agent']       = $proforma->sale_agent;
			$invoice['project_id']       = $proforma->project_id;
			$invoice['cycles']           = $proforma->cycles;
			$invoice['total_cycles']     = 0;
			// Since version 1.0.6
			$invoice['billing_street']   = clear_textarea_breaks($proforma->billing_street);
			$invoice['billing_city']     = $proforma->billing_city;
			$invoice['billing_state']    = $proforma->billing_state;
			$invoice['billing_zip']      = $proforma->billing_zip;
			$invoice['billing_country']  = $proforma->billing_country;
			$invoice['shipping_street']  = clear_textarea_breaks($proforma->shipping_street);
			$invoice['shipping_city']    = $proforma->shipping_city;
			$invoice['shipping_state']   = $proforma->shipping_state;
			$invoice['shipping_zip']     = $proforma->shipping_zip;
			$invoice['shipping_country'] = $proforma->shipping_country;
			if ($proforma->include_shipping == 1) {
				$invoice['include_shipping'] = $proforma->include_shipping;
			}
			$invoice['show_shipping_on_invoice'] = $proforma->show_shipping_on_invoice;
			// Set to unpaid status automatically
			$invoice['status']                = self::STATUS_UNPAID;
			$invoice['clientnote']            = $proforma->clientnote;
			$invoice['adminnote']             = $proforma->adminnote;
			$invoice['allowed_payment_modes'] = unserialize($proforma->allowed_payment_modes);

			$invoice['invoice_language']            = $proforma->invoice_language;
			$invoice['template']             = $proforma->template;
			$invoice['paper_format']            = $proforma->paper_format;
			$invoice['color']             = $proforma->color;

//pre($invoice);

			$invoice['newitems']              = [];
			$key                                       = 1;

			$custom_fields_items = get_custom_fields('items');
			foreach ($proforma->items as $item) {
				$invoice['newitems'][$key]['description']      = $item['description'];
				$invoice['newitems'][$key]['long_description'] = clear_textarea_breaks($item['long_description']);
				$invoice['newitems'][$key]['qty']              = $item['qty'];
				$invoice['newitems'][$key]['unit']             = $item['unit'];
				$invoice['newitems'][$key]['taxname']          = [];
				$taxes                                                  = get_proforma_item_taxes($item['id']);
				foreach ($taxes as $tax) {
					// tax name is in format TAX1|10.00
					array_push($invoice['newitems'][$key]['taxname'], $tax['taxname']);
				}
				$invoice['newitems'][$key]['rate']  = $item['rate'];
				$invoice['newitems'][$key]['order'] = $item['item_order'];

				$invoice['newitems'][$key]['item_id'] = $item['item_id'];
				$invoice['newitems'][$key]['price_converted'] = $item['price_converted'];
				$invoice['newitems'][$key]['currency_rate'] = $item['currency_rate'];
				$invoice['newitems'][$key]['currency'] = $item['currency'];
				$invoice['newitems'][$key]['sku'] = $item['sku'];



				foreach ($custom_fields_items as $cf) {
					$invoice['newitems'][$key]['custom_fields']['items'][$cf['id']] = get_custom_field_value($item['id'], $cf['id'], 'items', false);

					if (!defined('COPY_CUSTOM_FIELDS_LIKE_HANDLE_POST')) {
						define('COPY_CUSTOM_FIELDS_LIKE_HANDLE_POST', true);
					}
				}
				$key++;
			}

			//pre($invoice);
			$id = $this->invoices_model->add($invoice);
			if ($id) {
				$this->db->where('id', $id);
				$this->db->update(db_prefix() . 'invoices', [
					'cancel_overdue_reminders' => $proforma->cancel_overdue_reminders,
				]);

				$custom_fields = get_custom_fields('invoice');
				foreach ($custom_fields as $field) {
					$value = get_custom_field_value($proforma->id, $field['id'], 'invoice', false);
					if ($value == '') {
						continue;
					}
					$this->db->insert(db_prefix() . 'customfieldsvalues', [
						'relid'   => $id,
						'fieldid' => $field['id'],
						'fieldto' => 'invoice',
						'value'   => $value,
					]);
				}

				$tags = get_tags_in($proforma->id, 'invoice');
				handle_tags_save($tags, $id, 'invoice');

				log_activity('Convert proforma to invoice' . format_invoice_number($proforma->id));

				hooks()->do_action('proforma_invoice', ['copy_from' => $proforma->id, 'copy_id' => $id]);

				return $id;
			}

			return false;

		}












				public function getDefaultSerie() {
					$this->db->where('type', 'proforma');
					$this->db->where('default', 1);
					$row = $this->db->get(db_prefix() . 'billing_series')->row();

					return $row;

				}
		    public function get_seria($seria = null){
		      if($seria == null){
		        $this->db->where('type', 'proforma');
		        $this->db->where('default', 1);
		        $row = $this->db->get(db_prefix() . 'billing_series')->row_array();

		        return $row;
		      }
		      $this->db->where('type', 'proforma');
		      $this->db->where('name', $seria);
		      $row = $this->db->get(db_prefix() . 'billing_series')->row_array();

		      return $row;
		    }
		    public function get_series() {
					$this->db->where('type', 'proforma');
					$rows = $this->db->get(db_prefix() . 'billing_series')->result_array();

					return $rows;

				}



		    public function increment_next_number($seria = null){
		      $this->db->where('name', $seria);
					$this->db->where('type', 'proforma');
		      $this->db->set('current_number', 'current_number+1', false);
		      $this->db->update(db_prefix() . 'billing_series');
		    }
		    public function decrement_next_number($seria = null){
		      $this->db->where('name', $seria);
					$this->db->where('type', 'proforma');
		      $this->db->set('current_number', 'current_number-1', false);
		      $this->db->update(db_prefix() . 'billing_series');
		    }

		/**
		 * Add check item by serie
		 * we return item found with the given serie
		 */
		function checkSerie($serie){

			$this->db->where('prefix', $serie);
			$row = $this->db->get(db_prefix() . 'proformas')->row_array();

			if($row){
				return $row;
			} else {
				return null;
			}
		}


	}
