// Imports
import Utils from '../Utils.js';
import Itemable from './itemable.js';
import TemplateService from '../templates/templateService.js'

class ItemableService {

    constructor() {
        this.itemableDataSubject = new rxjs.BehaviorSubject([]);
        this.itemableData$ = this.itemableDataSubject.asObservable();

        this.items_template = admin_url + 'invoice_product_items/loadItemsListView';
    }

    fetchData() {
        // Simulating fetching data from an API or database
        const fetchedData = [
            {
                id: 1,
                rel_id: 1001,
                rel_type: 'order',
                description: 'Sample item 1',
                long_description: 'This is a sample item 1',
                qty: 5,
                rate: 10.99,
                unit: 'pcs',
                item_order: 1,
                type: 'regular',
                item_id: 101,
                price_converted: 54.96,
                currency_rate: 1.2345,
                currency: 1,
                sku: 'ABC123'
            },
            {
                id: 2,
                rel_id: 1002,
                rel_type: 'invoice',
                description: 'Sample item 2',
                long_description: 'This is a sample item 2',
                qty: 3,
                rate: 7.99,
                unit: 'pcs',
                item_order: 2,
                type: 'regular',
                item_id: 102,
                price_converted: 23.97,
                currency_rate: 1.2345,
                currency: 1,
                sku: 'DEF456'
            }
        ];

        // Transform the fetched data into Itemable instances
        const transformedData = fetchedData.map(data => new Itemable(data));

        // Emit the transformed data to subscribers
        this.itemableDataSubject.next(transformedData);
    }

    renderTemplate(item, index) {
      TemplateService.loadTemplate(admin_url + 'invoice_product_items/loadItemsListView').subscribe(
        template => {
          // Check if the loaded template is a string
          if (typeof template === 'string') {
            // Render the template with the item data
            const renderedTemplate = _.template(template)({ item, index });
    
            // Append the rendered template to the container
            const container = document.getElementById('invoice_itamable_items');
            container.insertAdjacentHTML('beforeend', renderedTemplate);
    
            // Check if the event listener has already been added
            if (!this.editModalEventListenerAdded) {
              this.bindEditModalEvent();
              this.editModalEventListenerAdded = true;
            }

            if (!this.deleteModalEventListenerAdded) {
              this.bindDeleteModalEvent();
              this.deleteModalEventListenerAdded = true;
            }
    
            // Further process the rendered template for the current item as needed
            // ...
          } else {
            console.error('Error: The loaded template is not a string.');
          }
        },
        error => {
          console.error('Error loading itemable_list.html template:', error);
        }
      );
    }

    getItemData(itemIndex) {
        // Use the item index to fetch the item data
        console.log(itemIndex);
        const itemData = this.itemableDataSubject.getValue()[itemIndex];
        console.log(itemData);
        // Return the item data
        return itemData;
    }

    addItemable(itemable) {
        // Add logic to save the itemable to the database or API
        // ...

        // Get the current itemable data
        const currentData = this.itemableDataSubject.getValue();

        // Update the itemable data by adding the new itemable
        const updatedData = [...currentData, itemable];

        // Emit the updated itemable data to subscribers
        this.itemableDataSubject.next(updatedData);
    }

    updateItemable(itemable) {
        // Add logic to update the itemable in the database or API
        // ...

        // Get the current itemable data
        const currentData = this.itemableDataSubject.getValue();

        // Find the index of the itemable to be updated
        const itemableIndex = currentData.findIndex(i => i.id === itemable.id);

        // If the itemable is found, update it in the data array
        if (itemableIndex > -1) {
            currentData[itemableIndex] = itemable;
            this.itemableDataSubject.next(currentData);
        }
    }

    deleteItemable(itemableId) {
        // Add logic to delete the itemable from the database or API
        // ...

        // Get the current itemable data
        const currentData = this.itemableDataSubject.getValue();

        // Filter out the deleted itemable from the data array
        const updatedData = currentData.filter(i => i.id !== itemableId);

        // Emit the updated itemable data to subscribers
        this.itemableDataSubject.next(updatedData);
    }

    async fetchInvoiceItems(invoiceID) {
        const url = admin_url + 'invoice_product_items/get_invoice_items';
      
        try {
          const response = await $.ajax({
            url: url,
            method: 'POST',
            dataType: 'json',
            data: { invoiceID: invoiceID },
          });
      
          // Transform the response data into Itemable instances
          const transformedData = response.map((data) => new Itemable(data));
      
          // Update the itemableDataSubject with the transformed data
          this.itemableDataSubject.next(transformedData);
      
          return transformedData;
        } catch (error) {
          // Handle the error
          const errorMessage = Utils.formatErrorMessage(error.statusText);
          throw errorMessage;
        }
      }

      bindEditModalEvent() {
        document.getElementById('invoice_itamable_items').addEventListener('click', (event) => {
          if (event.target.classList.contains('editInvoiceItem')) {
            // Get the item index from the clicked element
            const itemIndex = event.target.dataset.itemIndex;
            console.log(itemIndex);
            // Fetch the item data based on the itemIndex or any other logic
            const item = this.getItemData(itemIndex);
            console.log(item);
            // Render the itemable modal with the item data
            this.renderItemableModal(item);
          }
        });
      }


      bindDeleteModalEvent() {
        if (!this.deleteModalEventListenerAdded) {
          document.addEventListener('click', (event) => {
            if (event.target.classList.contains('deleteInvoiceItem')) {
              // Get the item index from the clicked element
              const itemIndex = event.target.getAttribute('data-item-index');
      
              // Fetch the item data based on the itemIndex or any other logic
              const item = this.getItemData(itemIndex);
      
              // Show the delete confirmation modal
              this.showDeleteConfirmationModal(item);
            }
          }, { once: true });
      
          this.deleteModalEventListenerAdded = true;
        }
      }
    

    async renderModal(url, container) {
        try {
            // Load the itemable modal template
            const template = await TemplateService.loadTemplate(url).toPromise();

            // Append the template to the modal container
            const modalContainer = document.getElementById(container);
            modalContainer.innerHTML = template;
        } catch (error) {
            console.error('Error loading itemable modal template:', error);
        }
    }

    async renderItemableDeleteModal() {
        try {
          // Render the delete modal form
          await this.renderModal(admin_url + 'invoice_product_items/loadItemDeleteModalView', 'invoice_itamable_itemsModalDetele_container');
        } catch (error) {
          console.error('Error rendering itemable delete modal:', error);
        }
      }

      showDeleteConfirmationModal(item) {
        // Render the delete modal
        this.renderItemableDeleteModal()
          .then(() => {
            // Populate the modal with the item data
            const deleteModalContainer = document.getElementById('invoice_itamable_itemsModalDetele_container');

            // Handle the delete button click event
            const deleteButton = deleteModalContainer.querySelector('#invoice_itamable_itemsModalEditDelete');
            deleteButton.addEventListener('click', () => {
                console.log(item);
              this.deleteItemable(item.id);
              $('#invoice_itamable_itemsModalDetele').modal('hide');
            });
      
            // Show the delete confirmation modal
            $('#invoice_itamable_itemsModalDetele').modal('show');
          })
          .catch((error) => {
            console.error('Error rendering itemable delete modal:', error);
          });
      }      

    async renderItemableModal(item) {
        console.log(item.description);
        if (!item) {
            console.error('Item not found');
            return;
        }

        // Render the itemable modal form
        await this.renderModal(admin_url + 'invoice_product_items/loadItemModalView', 'invoice_itamable_modal_container');

        // Populate the form fields with the item data
        const itemDescriptionInput = document.getElementById('item_description');
        itemDescriptionInput.value = item.description;

        const itemLongDescriptionInput = document.getElementById('item_long_description');
        itemLongDescriptionInput.value = item.long_description;

        const itemSkuInput = document.getElementById('item_sku');
        itemSkuInput.value = item.sku;

        const itemQtyInput = document.getElementById('item_qty');
        itemQtyInput.value = item.qty;

        const itemCotaInput = document.getElementById('item_cota');
        itemCotaInput.value = item.taxname;

        const itemCurrencyInput = document.getElementById('item_currency');
        itemCurrencyInput.value = item.currency_name;

        const itemPriceConvertedInput = document.getElementById('item_price_converted');
        itemPriceConvertedInput.value = item.price_converted;

        // Show the modal
        $('#invoice_itamable_modal').modal('show');
    }


}

export default ItemableService;
